/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.tiles;

import javax.annotation.Nonnull;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import forestry.api.core.climate.IClimatePosition;
import forestry.api.core.climate.IClimateRegion;
import forestry.api.multiblock.IGreenhouseComponent;
import forestry.api.multiblock.IMultiblockController;
import forestry.apiculture.network.packets.PacketActiveUpdate;
import forestry.core.proxy.Proxies;
import forestry.core.tiles.IActivatable;
import forestry.greenhouse.multiblock.IGreenhouseControllerInternal;

public class TileGreenhouseClimatiser extends TileGreenhouse implements IActivatable, IGreenhouseComponent.Climatiser {
	
	protected static final int WORK_CYCLES = 1;
	protected static final int ENERGY_PER_OPERATION = 150;
	
	protected enum ClimitiserType {
		TEMPERATURE, HUMIDITY
	}
	
	protected interface IClimitiserDefinition {
		float getChange();
		
		boolean isPositiv();
		
		int getClimitiseRange();
		
		ClimitiserType getType();
	}
	
	private final IClimitiserDefinition definition;
	
	protected EnumFacing inwards;
	protected EnumFacing leftwards;
	protected BlockPos maxPos;
	protected BlockPos minPos;
	
	protected int workingTime = 0;
	
	private boolean active;
	
	protected TileGreenhouseClimatiser(IClimitiserDefinition definition) {
		this.definition = definition;
	}
	
	@Override
	public void onMachineBroken() {
		inwards = null;
		leftwards = null;
		
		minPos = null;
		maxPos = null;
	}
	
	@Override
	public void onMachineAssembled(IMultiblockController multiblockController, BlockPos minCoord, BlockPos maxCoord) {
		recalculateDirections(minCoord, maxCoord);
		
		if(leftwards != null){
			maxPos = func_174877_v().func_177967_a(inwards, definition.getClimitiseRange() / 2).func_177967_a(leftwards, definition.getClimitiseRange() / 2).func_177967_a(EnumFacing.UP, definition.getClimitiseRange() / 2);
			minPos = func_174877_v().func_177972_a(inwards).func_177967_a(leftwards.func_176734_d(), definition.getClimitiseRange() / 2).func_177967_a(EnumFacing.DOWN, definition.getClimitiseRange() / 2);
			
		}else{
			maxPos = func_174877_v().func_177967_a(inwards, definition.getClimitiseRange() / 2).func_177967_a(EnumFacing.EAST, definition.getClimitiseRange() / 2).func_177967_a(EnumFacing.NORTH, definition.getClimitiseRange() / 2);
			minPos = func_174877_v().func_177972_a(inwards).func_177967_a(EnumFacing.WEST, definition.getClimitiseRange() / 2).func_177967_a(EnumFacing.SOUTH, definition.getClimitiseRange() / 2);
		}
	}
	
	@Override
	public int getTicksForChange(IClimateRegion region) {
		return 20;
	}
	
	@Override
	public void changeClimate(int tick, IClimateRegion region) {
		IMultiblockController controller = getMultiblockLogic().getController();
		if(getMultiblockLogic().isConnected() && controller != null && controller.isAssembled() && minPos != null && maxPos != null && region != null){
			IGreenhouseControllerInternal greenhouseInternal = (IGreenhouseControllerInternal) controller;
			boolean canWork = true;
			for (IGreenhouseComponent.Listener listenerComponent : greenhouseInternal.getListenerComponents()) {
				if(canWork){
					canWork = listenerComponent.getGreenhouseListener().canWork(greenhouseInternal, canWork);
				}
			}
			if (canWork && workingTime == 0 && greenhouseInternal.getEnergyManager().consumeEnergyToDoWork(WORK_CYCLES, ENERGY_PER_OPERATION)) {
				int dimensionID = field_145850_b.field_73011_w.getDimension();
				
				for(BlockPos pos : BlockPos.func_177980_a(maxPos, minPos)){
					IClimatePosition position = region.getPositions().get(pos);
					if(position != null){
						if (definition.getType() == ClimitiserType.TEMPERATURE) {
							if(definition.isPositiv()){
								if(position.getTemperature() >= 2.0F){
									if(position.getTemperature() > 2.0F){
										position.setTemperature(2.0F);
									}
									continue;
								}
							}else{
								if(position.getTemperature() <= 0.0F){
									if(position.getTemperature() < 0.0F){
										position.setTemperature(0.0F);
									}
									continue;
								}
							}
						}else{
							if(definition.isPositiv()){
								if(position.getHumidity() >= 2.0F){
									if(position.getHumidity() > 2.0F){
										position.setHumidity(2.0F);
									}
									continue;
								}
							}else{ 
								if(position.getHumidity() <= 0.0F){
									if(position.getHumidity() < 0.0F){
										position.setHumidity(0.0F);
									}
									continue;
								}
							}
						}
						
						double distance = pos.func_177951_i(this.field_174879_c);
						int maxDistance = definition.getClimitiseRange();
						if(distance <= maxDistance){
							if (definition.getType() == ClimitiserType.TEMPERATURE) {
								position.addTemperature((definition.isPositiv() ? +1 : -1) * ((distance == 0) ? definition.getChange() : (float) (definition.getChange() / distance)));
							}else{
								position.addHumidity((definition.isPositiv() ? +1 : -1) * ((distance == 0) ? definition.getChange() : (float) (definition.getChange() / distance)));
							}
						}
					}
				}
				
				// TODO: Add config entry for a time modifier and a energy modifier.
				// one tick of work for every 10 RF
				workingTime += ENERGY_PER_OPERATION / 25;
			}
	
			if (workingTime > 0) {
				workingTime--;
			}
	
			setActive(workingTime > 0);
		}else if(isActive()){
			setActive(false);
		}
	}
	
	/* LOADING & SAVING */
	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);
		workingTime = nbttagcompound.func_74762_e("Heating");
		setActive(workingTime > 0);
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);
		nbttagcompound.func_74768_a("Heating", workingTime);
		return nbttagcompound;
	}
	
	/* Network */
	@Override
	protected void encodeDescriptionPacket(NBTTagCompound packetData) {
		super.encodeDescriptionPacket(packetData);
		packetData.func_74757_a("Active", active);
	}

	@Override
	protected void decodeDescriptionPacket(NBTTagCompound packetData) {
		super.decodeDescriptionPacket(packetData);
		setActive(packetData.func_74767_n("Active"));
	}

	/* IActivatable */
	@Override
	public boolean isActive() {
		return active;
	}
	
	public void recalculateDirections(BlockPos minCoord, BlockPos maxCoord) {
		inwards = null;
		leftwards = null;

		int facesMatching = 0;
		if (maxCoord.func_177958_n() == func_174877_v().func_177958_n() || minCoord.func_177958_n() == func_174877_v().func_177958_n()) {
			facesMatching++;
		}
		if (maxCoord.func_177956_o() == func_174877_v().func_177956_o() || minCoord.func_177956_o() == func_174877_v().func_177956_o()) {
			facesMatching++;
		}
		if (maxCoord.func_177952_p() == func_174877_v().func_177952_p() || minCoord.func_177952_p() == func_174877_v().func_177952_p()) {
			facesMatching++;
		}
		if (facesMatching == 1) {
			if (maxCoord.func_177958_n() == func_174877_v().func_177958_n()) {
				inwards = EnumFacing.WEST;
				leftwards = EnumFacing.SOUTH;
			} else if (minCoord.func_177958_n() == func_174877_v().func_177958_n()) {
				inwards = EnumFacing.EAST;
				leftwards = EnumFacing.NORTH;
			} else if (maxCoord.func_177952_p() == func_174877_v().func_177952_p()) {
				inwards = EnumFacing.NORTH;
				leftwards = EnumFacing.WEST;
			} else if (minCoord.func_177952_p() == func_174877_v().func_177952_p()) {
				inwards = EnumFacing.SOUTH;
				leftwards = EnumFacing.EAST;
			} else if (maxCoord.func_177956_o() == func_174877_v().func_177956_o()) {
				inwards = EnumFacing.DOWN;
			} else {
				inwards = EnumFacing.UP;
			}
		}else{
			inwards = EnumFacing.DOWN;
		}
	}
	
	public IClimitiserDefinition getDefinition() {
		return definition;
	}

	@Override
	public void setActive(boolean active) {
		if (this.active == active) {
			return;
		}

		this.active = active;

		if (field_145850_b != null) {
			if (field_145850_b.field_72995_K) {
				field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
			} else {
				Proxies.net.sendNetworkPacket(new PacketActiveUpdate(this), field_145850_b);
			}
		}
	}

}
