/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.blocks;

import javax.annotation.Nonnull;
import java.util.EnumMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import net.minecraft.block.Block;
import net.minecraft.block.SoundType;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.block.model.ModelResourceLocation;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.IStringSerializable;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

import net.minecraftforge.client.model.ModelLoader;
import net.minecraftforge.common.property.ExtendedBlockState;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.common.property.IUnlistedProperty;
import net.minecraftforge.common.property.Properties;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.core.ForestryAPI;
import forestry.api.core.ICamouflagedTile;
import forestry.api.core.IModelManager;
import forestry.api.core.ISpriteRegister;
import forestry.api.core.ITextureManager;
import forestry.api.core.Tabs;
import forestry.api.multiblock.IGreenhouseComponent;
import forestry.core.CreativeTabForestry;
import forestry.core.blocks.BlockStructure;
import forestry.core.blocks.IColoredBlock;
import forestry.core.blocks.propertys.UnlistedBlockAccess;
import forestry.core.blocks.propertys.UnlistedBlockPos;
import forestry.core.utils.CamouflageUtil;
import forestry.greenhouse.tiles.TileGreenhouseControl;
import forestry.greenhouse.tiles.TileGreenhouseDoor;
import forestry.greenhouse.tiles.TileGreenhouseDryer;
import forestry.greenhouse.tiles.TileGreenhouseFan;
import forestry.greenhouse.tiles.TileGreenhouseGearbox;
import forestry.greenhouse.tiles.TileGreenhouseHatch;
import forestry.greenhouse.tiles.TileGreenhouseHeater;
import forestry.greenhouse.tiles.TileGreenhousePlain;
import forestry.greenhouse.tiles.TileGreenhouseSprinkler;
import forestry.greenhouse.tiles.TileGreenhouseValve;
import forestry.plugins.ForestryPluginUids;

public abstract class BlockGreenhouse extends BlockStructure implements ISpriteRegister, IColoredBlock {
	private static final AxisAlignedBB SPRINKLER_BOUNDS = new AxisAlignedBB(0.3125F, 0.25F, 0.3125F, 0.6875F, 1F, 0.6875F);
	public static final PropertyEnum<State> STATE = PropertyEnum.func_177709_a("state", State.class);
	
	public enum State implements IStringSerializable {
		ON, OFF;

		@Override
		public String func_176610_l() {
			return name().toLowerCase(Locale.ENGLISH);
		}
	}
	
	public static Map<BlockGreenhouseType, BlockGreenhouse> create() {
		Map<BlockGreenhouseType, BlockGreenhouse> blockMap = new EnumMap<>(BlockGreenhouseType.class);
		for (final BlockGreenhouseType type : BlockGreenhouseType.VALUES) {
			if (type == BlockGreenhouseType.BUTTERFLY_HATCH) {
				if (!ForestryAPI.enabledPlugins.contains(ForestryPluginUids.LEPIDOPTEROLOGY)) {
					continue;
				}
			}
			
			BlockGreenhouse block;
			if (type == BlockGreenhouseType.DOOR) {
				block = new BlockGreenhouseDoor();
			} else {
				block = new BlockGreenhouse() {
					@Nonnull
					@Override
					public BlockGreenhouseType getGreenhouseType() {
						return type;
					}
					
					@Nonnull
					@Override
					public SoundType func_185467_w() {
						if(type == BlockGreenhouseType.SPRINKLER || type == BlockGreenhouseType.GLASS){
							return SoundType.field_185853_f;
						}
						return super.func_185467_w();
					}
				};
			}
			blockMap.put(type, block);
		}
		return blockMap;
	}
	
	public BlockGreenhouse() {
		super(Material.field_151576_e);
		BlockGreenhouseType greenhouseType = getGreenhouseType();
		IBlockState defaultState = this.field_176227_L.func_177621_b();
		if (greenhouseType.activatable && greenhouseType != BlockGreenhouseType.SPRINKLER) {
			defaultState = defaultState.func_177226_a(STATE, State.OFF);
		}
		func_180632_j(defaultState);
		
		func_149711_c(1.0f);
		setHarvestLevel("pickaxe", 0);
		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.FARMING)) {
			func_149647_a(Tabs.tabAgriculture);
		} else {
			func_149647_a(CreativeTabForestry.tabForestry);
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public AxisAlignedBB func_180640_a(IBlockState state, World worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return SPRINKLER_BOUNDS.func_186670_a(pos);
		}
		return super.func_180640_a(state, worldIn, pos);
	}

	@Override
	public AxisAlignedBB func_180646_a(IBlockState blockState, World worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return SPRINKLER_BOUNDS;
		}
		return super.func_180646_a(blockState, worldIn, pos);
	}

	@Override
	public int func_176201_c(IBlockState state) {
		return 0;
	}

	@Override
	public IBlockState getExtendedState(IBlockState state, IBlockAccess world, BlockPos pos) {
		if (getGreenhouseType() != BlockGreenhouseType.SPRINKLER) {
			return ((IExtendedBlockState) super.getExtendedState(state, world, pos)).withProperty(UnlistedBlockPos.POS, pos)
					.withProperty(UnlistedBlockAccess.BLOCKACCESS, world);
		}
		return super.getExtendedState(state, world, pos);
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return new ExtendedBlockState(this, new IProperty[]{Properties.StaticProperty}, new IUnlistedProperty[]{Properties.AnimationProperty});
		} else if (getGreenhouseType().activatable) {
			return new ExtendedBlockState(this, new IProperty[]{STATE}, new IUnlistedProperty[]{UnlistedBlockPos.POS, UnlistedBlockAccess.BLOCKACCESS});
		} else {
			return new ExtendedBlockState(this, new IProperty[0], new IUnlistedProperty[]{UnlistedBlockPos.POS, UnlistedBlockAccess.BLOCKACCESS});
		}
	}
	
	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return state.func_177226_a(Properties.StaticProperty, true);
		} else {
			return super.func_176221_a(state, worldIn, pos);
		}
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public void func_149666_a(Item item, CreativeTabs tab, List<ItemStack> list) {
		list.add(new ItemStack(item));
	}

	@Override
	public TileEntity func_149915_a(World world, int meta) {
		BlockGreenhouseType type = getGreenhouseType();
		switch (type) {
			case GEARBOX:
				return new TileGreenhouseGearbox();
			case SPRINKLER:
				return new TileGreenhouseSprinkler();
			case DRYER:
				return new TileGreenhouseDryer();
			case VALVE:
				return new TileGreenhouseValve();
			case FAN:
				return new TileGreenhouseFan();
			case HEATER:
				return new TileGreenhouseHeater();
			case DOOR:
				return new TileGreenhouseDoor();
			case CONTROL:
				return new TileGreenhouseControl();
			case HATCH_INPUT:
			case HATCH_OUTPUT:
				return new TileGreenhouseHatch();
			case BUTTERFLY_HATCH:
				return new TileGreenhouseHatch();
			default:
				return new TileGreenhousePlain();
		}
	}

	@Override
	public int colorMultiplier(IBlockState state, IBlockAccess worldIn, BlockPos pos, int tintIndex) {
		if(pos == null){
			return 0xffffff;
		}
		TileEntity tile = worldIn.func_175625_s(pos);
		if (tile instanceof ICamouflagedTile) {
			ItemStack camouflageStack = CamouflageUtil.getCamouflageBlock(worldIn, pos);

			if (tintIndex < 100 && camouflageStack != null) {
				Block block = Block.func_149634_a(camouflageStack.func_77973_b());
				if(block != null){
					IBlockState camouflageState = block.func_176203_a(camouflageStack.func_77952_i());
					
					int color = Minecraft.func_71410_x().func_184125_al().func_186724_a(camouflageState, worldIn, pos, tintIndex);
					if(color != -1){
						return color;
					}
				}
			}
		}

		return 0xffffff;
	}

	/* MODELS */
	@Override
	@SideOnly(Side.CLIENT)
	public BlockRenderLayer func_180664_k() {
		if (getGreenhouseType() == BlockGreenhouseType.GLASS || getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return BlockRenderLayer.TRANSLUCENT;
		}
		return BlockRenderLayer.CUTOUT;
	}

	@Override
	public boolean func_149686_d(IBlockState state) {
		return getGreenhouseType() != BlockGreenhouseType.GLASS && getGreenhouseType() != BlockGreenhouseType.SPRINKLER;
	}

	@Override
	public boolean func_149662_c(IBlockState state) {
		return getGreenhouseType() != BlockGreenhouseType.GLASS && getGreenhouseType() != BlockGreenhouseType.SPRINKLER;
	}
	
	@Override
	public boolean func_176196_c(World worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			if (worldIn.func_180495_p(pos.func_177984_a()).func_177230_c() == this) {
				return false;
			}
			if (!(worldIn.func_175625_s(pos.func_177984_a()) instanceof IGreenhouseComponent)) {
				return false;
			}
		}
		return super.func_176196_c(worldIn, pos);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public boolean func_176225_a(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		IBlockState iblockstate = blockAccess.func_180495_p(pos);
		Block block = iblockstate.func_177230_c();

		if (getGreenhouseType() == BlockGreenhouseType.GLASS) {
			if (blockAccess.func_180495_p(pos.func_177972_a(side)) != iblockstate) {
				return true;
			}

			if (block == this) {
				return false;
			}
			return block != this && super.func_176225_a(blockState, blockAccess, pos, side);
		} else if (getGreenhouseType() == BlockGreenhouseType.DOOR) {
			return super.func_176225_a(blockState, blockAccess, pos, side);
		}

		return super.func_176225_a(blockState, blockAccess, pos, side);
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public void registerModel(Item item, IModelManager manager) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			ModelLoader.setCustomModelResourceLocation(item, 0, new ModelResourceLocation("forestry:greenhouse.sprinkler", "inventory"));
		} else {
			ModelLoader.setCustomModelResourceLocation(item, 0, new ModelResourceLocation("forestry:greenhouse", "inventory"));
		}
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public void registerSprites(ITextureManager manager) {
		BlockGreenhouseType.registerSprites();
	}

	@Override
	public boolean canConnectRedstone(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side) {
		return getGreenhouseType() == BlockGreenhouseType.CONTROL;
	}
	
	@Nonnull
	public abstract BlockGreenhouseType getGreenhouseType();
}
