/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.lepidopterology.tiles;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;

import forestry.core.owner.IOwnedTile;
import forestry.core.owner.IOwnerHandler;
import forestry.core.owner.OwnerHandler;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import forestry.api.genetics.IAllele;
import forestry.api.lepidopterology.ButterflyManager;
import forestry.api.lepidopterology.IButterfly;
import forestry.api.lepidopterology.IButterflyCocoon;
import forestry.api.lepidopterology.IButterflyGenome;
import forestry.api.lepidopterology.IButterflyNursery;
import forestry.api.multiblock.IGreenhouseComponent;
import forestry.core.network.DataInputStreamForestry;
import forestry.core.network.DataOutputStreamForestry;
import forestry.core.network.IStreamable;
import forestry.core.network.packets.PacketTileStream;
import forestry.core.proxy.Proxies;
import forestry.core.utils.ItemStackUtil;
import forestry.core.utils.Log;
import forestry.core.utils.NBTUtilForestry;
import forestry.greenhouse.multiblock.GreenhouseController;
import forestry.lepidopterology.genetics.Butterfly;
import forestry.lepidopterology.genetics.ButterflyDefinition;

public class TileCocoon extends TileEntity implements IStreamable, IOwnedTile, IButterflyCocoon {
	private final OwnerHandler ownerHandler = new OwnerHandler();
	private int age;
	private int maturationTime;
	private IButterfly caterpillar = ButterflyDefinition.CabbageWhite.getIndividual();
	private BlockPos nursery;
	private boolean isSolid;

	public TileCocoon() {
	}
	
	public TileCocoon(boolean isSolid) {
		this.isSolid = isSolid;
		if (isSolid) {
			this.age = 2;
		}
	}

	/* SAVING & LOADING */
	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);

		if (nbttagcompound.func_74764_b("Caterpillar")) {
			caterpillar = new Butterfly(nbttagcompound.func_74775_l("Caterpillar"));
		}
		ownerHandler.readFromNBT(nbttagcompound);
		if (nbttagcompound.func_74764_b("nursery")) {
			NBTTagCompound nbt = nbttagcompound.func_74775_l("nursery");
			int x = nbt.func_74762_e("x");
			int y = nbt.func_74762_e("y");
			int z = nbt.func_74762_e("z");
			nursery = new BlockPos(x, y, z);
		}
		age = nbttagcompound.func_74762_e("Age");
		maturationTime = nbttagcompound.func_74762_e("CATMAT");
		isSolid = nbttagcompound.func_74767_n("isSolid");
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);

		if (caterpillar != null) {
			NBTTagCompound subcompound = new NBTTagCompound();
			caterpillar.writeToNBT(subcompound);
			nbttagcompound.func_74782_a("Caterpillar", subcompound);
		}
		ownerHandler.writeToNBT(nbttagcompound);
		if (nursery != null) {
			BlockPos pos = nursery;
			NBTTagCompound nbt = new NBTTagCompound();
			nbt.func_74768_a("x", pos.func_177958_n());
			nbt.func_74768_a("y", pos.func_177956_o());
			nbt.func_74768_a("z", pos.func_177952_p());
			nbttagcompound.func_74782_a("nursery", nbt);
		}
		nbttagcompound.func_74768_a("Age", age);
		nbttagcompound.func_74768_a("CATMAT", maturationTime);
		nbttagcompound.func_74757_a("isSolid", isSolid);
		return nbttagcompound;
	}

	@Override
	public void writeData(DataOutputStreamForestry data) throws IOException {
		String speciesUID = "";
		IButterfly caterpillar = getCaterpillar();
		if (caterpillar != null) {
			speciesUID = caterpillar.getIdent();
		}
		data.writeUTF(speciesUID);
		data.writeInt(age);
	}

	@Override
	public void readData(DataInputStreamForestry data) throws IOException {
		String speciesUID = data.readUTF();
		IButterfly caterpillar = getButterfly(speciesUID);
		setCaterpillar(caterpillar);
		age = data.readInt();
	}

	private static IButterfly getButterfly(String speciesUID) {
		IAllele[] butterflyTemplate = ButterflyManager.butterflyRoot.getTemplate(speciesUID);
		if (butterflyTemplate == null) {
			return null;
		}
		return ButterflyManager.butterflyRoot.templateAsIndividual(butterflyTemplate);
	}

	@Override
	public IOwnerHandler getOwnerHandler() {
		return ownerHandler;
	}

	@Override
	public boolean shouldRefresh(World world, BlockPos pos, IBlockState oldState, IBlockState newSate) {
		return !Block.func_149680_a(oldState.func_177230_c(), newSate.func_177230_c());
	}

	/* INETWORKEDENTITY */
	@Nullable
	@Override
	public SPacketUpdateTileEntity func_189518_D_() {
		return new SPacketUpdateTileEntity(this.func_174877_v(), 0, func_189517_E_());
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189517_E_() {
		NBTTagCompound tag = super.func_189517_E_();
		return NBTUtilForestry.writeStreamableToNbt(this, tag);
	}

	@Override
	public void handleUpdateTag(@Nonnull NBTTagCompound tag) {
		super.handleUpdateTag(tag);
		NBTUtilForestry.readStreamableFromNbt(this, tag);
	}
	
	@Override
	public void onDataPacket(NetworkManager net, SPacketUpdateTileEntity pkt) {
		super.onDataPacket(net, pkt);
		NBTTagCompound nbt = pkt.func_148857_g();
		handleUpdateTag(nbt);
	}

	public void onBlockTick() {
		if (caterpillar == null) {
			field_145850_b.func_175698_g(func_174877_v());
			return;
		}

		maturationTime++;

		IButterflyGenome caterpillarGenome = caterpillar.getGenome();
		int caterpillarMatureTime = Math.round((float) caterpillarGenome.getLifespan() / (caterpillarGenome.getFertility() * 2));

		if (maturationTime >= caterpillarMatureTime) {
			if (age < 2) {
				age++;
				maturationTime = 0;
			} else if (caterpillar.canTakeFlight(field_145850_b, func_174877_v().func_177958_n(), func_174877_v().func_177956_o(), func_174877_v().func_177952_p())) {
				IGreenhouseComponent.ButterflyHatch hatch = GreenhouseController.getGreenhouseButterflyHatch(field_145850_b, field_174879_c);
				ItemStack[] cocoonDrops;
				if (hatch != null) {
					cocoonDrops = hatch.addCocoonLoot(this);
				} else {
					cocoonDrops = caterpillar.getCocoonDrop(this);
				}
				for (ItemStack drop : cocoonDrops) {
					ItemStackUtil.dropItemStackAsEntity(drop, field_145850_b, field_174879_c);
				}
				field_145850_b.func_175698_g(func_174877_v());
				attemptButterflySpawn(field_145850_b, caterpillar, func_174877_v());
			}
		}
	}
	
	private static void attemptButterflySpawn(World world, IButterfly butterfly, BlockPos pos) {
		if (ButterflyManager.butterflyRoot.spawnButterflyInWorld(world, butterfly.copy(), pos.func_177958_n(), pos.func_177956_o() + 0.1f, pos.func_177952_p()) != null) {
			Log.trace("A caterpillar '%s' hatched at %s/%s/%s.", butterfly.getDisplayName(), pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
		}
	}

	@Override
	public BlockPos getCoordinates() {
		return func_174877_v();
	}

	@Override
	public IButterfly getCaterpillar() {
		return caterpillar;
	}

	@Override
	public IButterflyNursery getNursery() {
		TileEntity nursery = field_145850_b.func_175625_s(this.nursery);
		if (nursery instanceof IButterflyNursery) {
			return (IButterflyNursery) nursery;
		}
		return null;
	}
	
	@Override
	public void setNursery(IButterflyNursery nursery) {
		this.nursery = nursery.getCoordinates();
	}

	@Override
	public void setCaterpillar(IButterfly butterfly) {
		this.caterpillar = butterfly;
		sendNetworkUpdate();
	}

	private void sendNetworkUpdate() {
		Proxies.net.sendNetworkPacket(new PacketTileStream(this), field_145850_b);
	}
	
	public int getAge() {
		return age;
	}
	
	public void setAge(int age) {
		this.age = age;
	}
	
	public ItemStack[] getCocoonDrops() {
		return caterpillar.getCocoonDrop(this);
	}

	@Override
	public boolean isSolid() {
		return isSolid;
	}

}
