/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.utils;

import java.util.List;

import forestry.core.gui.slots.SlotCrafter;
import forestry.core.gui.slots.SlotForestry;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.ClickType;
import net.minecraft.inventory.Slot;
import net.minecraft.inventory.SlotCrafting;
import net.minecraft.item.ItemStack;

public abstract class SlotUtil {

	public static boolean isSlotInRange(int slotIndex, int start, int count) {
		return slotIndex >= start && slotIndex < start + count;
	}

	public static ItemStack slotClickPhantom(SlotForestry slot, int mouseButton, ClickType clickTypeIn, EntityPlayer player) {
		ItemStack stack = null;

		ItemStack stackSlot = slot.func_75211_c();
		if (stackSlot != null) {
			stack = stackSlot.func_77946_l();
		}

		if (mouseButton == 2) {
			fillPhantomSlot(slot, null, mouseButton);
		} else if (mouseButton == 0 || mouseButton == 1) {
			InventoryPlayer playerInv = player.field_71071_by;

			ItemStack stackHeld = playerInv.func_70445_o();

			if (stackSlot == null) {
				if (stackHeld != null && slot.func_75214_a(stackHeld)) {
					fillPhantomSlot(slot, stackHeld, mouseButton);
				}
			} else if (stackHeld == null) {
				adjustPhantomSlot(slot, mouseButton, clickTypeIn);
			} else if (slot.func_75214_a(stackHeld)) {
				if (ItemStackUtil.isIdenticalItem(stackSlot, stackHeld)) {
					adjustPhantomSlot(slot, mouseButton, clickTypeIn);
				} else {
					fillPhantomSlot(slot, stackHeld, mouseButton);
				}
			}
		} else if (mouseButton == 5) {
			InventoryPlayer playerInv = player.field_71071_by;
			ItemStack stackHeld = playerInv.func_70445_o();
			if (!slot.func_75216_d()) {
				fillPhantomSlot(slot, stackHeld, mouseButton);
			}
		}
		return stack;
	}

	public static ItemStack transferStackInSlot(List inventorySlots, EntityPlayer player, int slotIndex) {
		Slot slot = (Slot) inventorySlots.get(slotIndex);
		if (slot == null || !slot.func_75216_d()) {
			return null;
		}

		boolean fromCraftingSlot = slot instanceof SlotCrafting || slot instanceof SlotCrafter;

		int numSlots = inventorySlots.size();
		ItemStack stackInSlot = slot.func_75211_c();
		ItemStack originalStack = stackInSlot.func_77946_l();

		if (!shiftItemStack(inventorySlots, stackInSlot, slotIndex, numSlots, fromCraftingSlot)) {
			return null;
		}

		slot.func_75220_a(stackInSlot, originalStack);
		if (stackInSlot.field_77994_a <= 0) {
			slot.func_75215_d(null);
		} else {
			slot.func_75218_e();
		}

		if (stackInSlot.field_77994_a == originalStack.field_77994_a) {
			return null;
		}

		slot.func_82870_a(player, stackInSlot);
		return originalStack;
	}

	private static boolean shiftItemStack(List inventorySlots, ItemStack stackInSlot, int slotIndex, int numSlots, boolean fromCraftingSlot) {
		if (isInPlayerInventory(slotIndex)) {
			if (shiftToMachineInventory(inventorySlots, stackInSlot, numSlots)) {
				return true;
			}

			if (isInPlayerHotbar(slotIndex)) {
				return shiftToPlayerInventoryNoHotbar(inventorySlots, stackInSlot);
			} else {
				return shiftToHotbar(inventorySlots, stackInSlot);
			}
		} else {
			if (fromCraftingSlot) {
				if (shiftToMachineInventory(inventorySlots, stackInSlot, numSlots)) {
					return true;
				}
			}
			return shiftToPlayerInventory(inventorySlots, stackInSlot);
		}
	}

	private static void adjustPhantomSlot(SlotForestry slot, int mouseButton, ClickType clickTypeIn) {
		if (!slot.canAdjustPhantom()) {
			return;
		}
		ItemStack stackSlot = slot.func_75211_c();
		int stackSize;
		if (clickTypeIn == ClickType.QUICK_MOVE) {
			stackSize = mouseButton == 0 ? (stackSlot.field_77994_a + 1) / 2 : stackSlot.field_77994_a * 2;
		} else {
			stackSize = mouseButton == 0 ? stackSlot.field_77994_a - 1 : stackSlot.field_77994_a + 1;
		}

		if (stackSize > slot.func_75219_a()) {
			stackSize = slot.func_75219_a();
		}

		stackSlot.field_77994_a = stackSize;

		if (stackSlot.field_77994_a <= 0) {
			stackSlot = null;
		}

		slot.func_75215_d(stackSlot);
	}

	private static void fillPhantomSlot(SlotForestry slot, ItemStack stackHeld, int mouseButton) {
		if (!slot.canAdjustPhantom()) {
			return;
		}

		if (stackHeld == null) {
			slot.func_75215_d(null);
			return;
		}

		int stackSize = mouseButton == 0 ? stackHeld.field_77994_a : 1;
		if (stackSize > slot.func_75219_a()) {
			stackSize = slot.func_75219_a();
		}
		ItemStack phantomStack = stackHeld.func_77946_l();
		phantomStack.field_77994_a = stackSize;

		slot.func_75215_d(phantomStack);
	}

	private static boolean shiftItemStackToRange(List inventorySlots, ItemStack stackToShift, int start, int count) {
		boolean changed = shiftItemStackToRangeMerge(inventorySlots, stackToShift, start, count);
		changed |= shiftItemStackToRangeOpenSlots(inventorySlots, stackToShift, start, count);
		return changed;
	}

	private static boolean shiftItemStackToRangeMerge(List inventorySlots, ItemStack stackToShift, int start, int count) {
		if (stackToShift == null || !stackToShift.func_77985_e() || stackToShift.field_77994_a <= 0) {
			return false;
		}

		boolean changed = false;
		for (int slotIndex = start; stackToShift.field_77994_a > 0 && slotIndex < start + count; slotIndex++) {
			Slot slot = (Slot) inventorySlots.get(slotIndex);
			ItemStack stackInSlot = slot.func_75211_c();
			if (stackInSlot != null && ItemStackUtil.isIdenticalItem(stackInSlot, stackToShift)) {
				int resultingStackSize = stackInSlot.field_77994_a + stackToShift.field_77994_a;
				int max = Math.min(stackToShift.func_77976_d(), slot.func_75219_a());
				if (resultingStackSize <= max) {
					stackToShift.field_77994_a = 0;
					stackInSlot.field_77994_a = resultingStackSize;
					slot.func_75218_e();
					changed = true;
				} else if (stackInSlot.field_77994_a < max) {
					stackToShift.field_77994_a -= max - stackInSlot.field_77994_a;
					stackInSlot.field_77994_a = max;
					slot.func_75218_e();
					changed = true;
				}
			}
		}
		return changed;
	}

	private static boolean shiftItemStackToRangeOpenSlots(List inventorySlots, ItemStack stackToShift, int start, int count) {
		if (stackToShift == null || stackToShift.field_77994_a <= 0) {
			return false;
		}

		boolean changed = false;
		for (int slotIndex = start; stackToShift.field_77994_a > 0 && slotIndex < start + count; slotIndex++) {
			Slot slot = (Slot) inventorySlots.get(slotIndex);
			ItemStack stackInSlot = slot.func_75211_c();
			if (stackInSlot == null) {
				int max = Math.min(stackToShift.func_77976_d(), slot.func_75219_a());
				stackInSlot = stackToShift.func_77946_l();
				stackInSlot.field_77994_a = Math.min(stackToShift.field_77994_a, max);
				stackToShift.field_77994_a -= stackInSlot.field_77994_a;
				slot.func_75215_d(stackInSlot);
				slot.func_75218_e();
				changed = true;
			}
		}
		return changed;
	}

	private static final int playerInventorySize = 9 * 4;
	private static final int playerHotbarSize = 9;

	private static boolean isInPlayerInventory(int slotIndex) {
		return slotIndex < playerInventorySize;
	}

	private static boolean isInPlayerHotbar(int slotIndex) {
		return SlotUtil.isSlotInRange(slotIndex, playerInventorySize - playerHotbarSize, playerInventorySize);
	}

	private static boolean shiftToPlayerInventory(List inventorySlots, ItemStack stackInSlot) {
		int playerHotbarStart = playerInventorySize - playerHotbarSize;

		// try to merge with existing stacks, hotbar first
		boolean shifted = shiftItemStackToRangeMerge(inventorySlots, stackInSlot, playerHotbarStart, playerHotbarSize);
		shifted |= shiftItemStackToRangeMerge(inventorySlots, stackInSlot, 0, playerHotbarStart);

		// shift to open slots, hotbar first
		shifted |= shiftItemStackToRangeOpenSlots(inventorySlots, stackInSlot, playerHotbarStart, playerHotbarSize);
		shifted |= shiftItemStackToRangeOpenSlots(inventorySlots, stackInSlot, 0, playerHotbarStart);
		return shifted;
	}

	private static boolean shiftToPlayerInventoryNoHotbar(List inventorySlots, ItemStack stackInSlot) {
		int playerHotbarStart = playerInventorySize - playerHotbarSize;
		return shiftItemStackToRange(inventorySlots, stackInSlot, 0, playerHotbarStart);
	}

	private static boolean shiftToHotbar(List inventorySlots, ItemStack stackInSlot) {
		int playerHotbarStart = playerInventorySize - playerHotbarSize;
		return shiftItemStackToRange(inventorySlots, stackInSlot, playerHotbarStart, playerHotbarSize);
	}

	private static boolean shiftToMachineInventory(List inventorySlots, ItemStack stackToShift, int numSlots) {
		boolean success = false;
		if (stackToShift.func_77985_e()) {
			success = shiftToMachineInventory(inventorySlots, stackToShift, numSlots, true);
		}
		if (stackToShift.field_77994_a > 0) {
			success |= shiftToMachineInventory(inventorySlots, stackToShift, numSlots, false);
		}
		return success;
	}

	// if mergeOnly = true, don't shift into empty slots.
	private static boolean shiftToMachineInventory(List inventorySlots, ItemStack stackToShift, int numSlots, boolean mergeOnly) {
		for (int machineIndex = playerInventorySize; machineIndex < numSlots; machineIndex++) {
			Slot slot = (Slot) inventorySlots.get(machineIndex);
			if (mergeOnly && slot.func_75211_c() == null) {
				continue;
			}
			if (slot instanceof SlotForestry) {
				SlotForestry slotForestry = (SlotForestry) slot;
				if (!slotForestry.canShift() || slotForestry.isPhantom()) {
					continue;
				}
			}
			if (!slot.func_75214_a(stackToShift)) {
				continue;
			}
			if (shiftItemStackToRange(inventorySlots, stackToShift, machineIndex, 1)) {
				return true;
			}
		}
		return false;
	}
}
