/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.arboriculture.proxy;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.block.model.IBakedModel;
import net.minecraft.client.renderer.block.model.ModelResourceLocation;
import net.minecraft.client.renderer.block.model.multipart.Multipart;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.registry.IRegistry;
import net.minecraft.world.ColorizerFoliage;
import net.minecraftforge.client.event.ModelBakeEvent;
import net.minecraftforge.client.model.IModel;
import net.minecraftforge.client.model.ModelLoaderRegistry;
import net.minecraftforge.fml.client.registry.ClientRegistry;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import forestry.api.arboriculture.EnumPileType;
import forestry.api.arboriculture.IWoodItemMeshDefinition;
import forestry.api.arboriculture.IWoodStateMapper;
import forestry.api.arboriculture.IWoodType;
import forestry.api.arboriculture.WoodBlockKind;
import forestry.arboriculture.IWoodTyped;
import forestry.arboriculture.PluginArboriculture;
import forestry.arboriculture.blocks.BlockDecorativeLeaves;
import forestry.arboriculture.blocks.slab.BlockArbSlab;
import forestry.arboriculture.models.ModelDecorativeLeaves;
import forestry.arboriculture.models.ModelLeaves;
import forestry.arboriculture.models.ModelWoodPile;
import forestry.arboriculture.models.MultipartModel;
import forestry.arboriculture.models.WoodModelLoader;
import forestry.arboriculture.models.WoodTextureManager;
import forestry.arboriculture.render.CharcoalPileRenderer;
import forestry.arboriculture.tiles.TilePile;
import forestry.core.models.BlockModelEntry;
import forestry.core.models.ModelManager;
import forestry.core.models.SimpleRetexturedModel;
import forestry.core.models.WoodModelEntry;
import forestry.core.proxy.Proxies;

public class ProxyArboricultureClient extends ProxyArboriculture {
	private static Set<WoodModelEntry> woodModelEntrys = new HashSet<>();
	private static final Map<IWoodTyped, IWoodStateMapper> stateMappers = Maps.newIdentityHashMap();
	private static final Map<Item, IWoodItemMeshDefinition> shapers = Maps.newHashMap();
	
	@Override
	public void initializeModels() {
		{
			ModelResourceLocation blockModelLocation = new ModelResourceLocation("forestry:leaves");
			ModelResourceLocation itemModelLocation = new ModelResourceLocation("forestry:leaves", "inventory");
			BlockModelEntry blockModelIndex = new BlockModelEntry(blockModelLocation, itemModelLocation, new ModelLeaves(), PluginArboriculture.blocks.leaves);
			Proxies.render.registerBlockModel(blockModelIndex);
		}

		for (BlockDecorativeLeaves leaves : PluginArboriculture.blocks.leavesDecorative) {
			String resourceName = "forestry:leaves.decorative." + leaves.getBlockNumber();
			ModelResourceLocation blockModelLocation = new ModelResourceLocation(resourceName);
			ModelResourceLocation itemModeLocation = new ModelResourceLocation(resourceName, "inventory");
			BlockModelEntry blockModelIndex = new BlockModelEntry(blockModelLocation, itemModeLocation, new ModelDecorativeLeaves(), leaves);
			Proxies.render.registerBlockModel(blockModelIndex);
		}
		
		{
			ModelResourceLocation blockModelLocation = new ModelResourceLocation("forestry:pile", "type=wood");
			ModelResourceLocation itemModelLocation = new ModelResourceLocation("forestry:woodPile", "inventory");
			BlockModelEntry blockModelIndex = new BlockModelEntry(blockModelLocation, itemModelLocation, new ModelWoodPile(), PluginArboriculture.blocks.piles.get(EnumPileType.WOOD));
			Proxies.render.registerBlockModel(blockModelIndex);
		}
		
		ClientRegistry.bindTileEntitySpecialRenderer(TilePile.class, new CharcoalPileRenderer());
		ModelLoaderRegistry.registerLoader(WoodModelLoader.INSTANCE);
		for(BlockArbSlab slab : PluginArboriculture.blocks.slabsDouble){
			registerWoodModel(slab, true);
		}
		for(BlockArbSlab slab : PluginArboriculture.blocks.slabsDoubleFireproof){
			registerWoodModel(slab, true);
		}
	}
	
	public static void registerWoodMeshDefinition(Item item, IWoodItemMeshDefinition definition) {
		ModelManager.getInstance().registerItemModel(item, definition);
		shapers.put(item, definition);
	}

	public static void registerWoodStateMapper(Block block, IWoodStateMapper stateMapper) {
		if (block instanceof IWoodTyped) {
			IWoodTyped woodTyped = (IWoodTyped) block;
			Proxies.render.registerStateMapper(block, stateMapper);
			stateMappers.put(woodTyped, stateMapper);
		}
	}

	@SubscribeEvent
	public <T extends Block & IWoodTyped> void onModelBake(ModelBakeEvent event) {
		WoodModelLoader.INSTANCE.isEnabled = true;
		IRegistry<ModelResourceLocation, IBakedModel> registry = event.getModelRegistry();

		for (WoodModelEntry<T> entry : woodModelEntrys) {
			T woodTyped = entry.woodTyped;
			WoodBlockKind woodKind = woodTyped.getBlockKind();
			IWoodStateMapper woodMapper = stateMappers.get(woodTyped);

			for (IBlockState blockState : woodTyped.func_176194_O().func_177619_a()) {
				IWoodType woodType;
				ItemStack itemStack;
				if (entry.withVariants) {
					int meta = woodTyped.func_176201_c(blockState);
					woodType = woodTyped.getWoodType(meta);
					itemStack = new ItemStack(woodTyped, 1, meta);
				} else {
					woodType = woodTyped.getWoodType(0);
					itemStack = new ItemStack(woodTyped);
				}
				IWoodItemMeshDefinition definition = shapers.get(itemStack.func_77973_b());
				ImmutableMap<String, String> textures = WoodTextureManager.getTextures(woodType, woodKind);
				if (definition != null) {
					retextureItemModel(registry, textures, woodType, woodKind, itemStack, definition);
				}
				if (woodMapper != null) {
					retexturBlockModel(registry, textures, woodType, woodKind, blockState, woodMapper);
				}
			}
		}
	}

	private void retextureItemModel(IRegistry<ModelResourceLocation, IBakedModel> registry,
			ImmutableMap<String, String> textures, IWoodType woodType, WoodBlockKind woodKind, ItemStack itemStack,
			IWoodItemMeshDefinition woodDefinition) {
		if (woodKind != WoodBlockKind.DOOR) {
			IModel basicItemModel = ModelLoaderRegistry
					.getModelOrMissing(woodDefinition.getDefaultModelLocation(itemStack));
			ModelResourceLocation basicItemLocation = woodDefinition.func_178113_a(itemStack);

			registry.func_82595_a(basicItemLocation,
					new SimpleRetexturedModel(woodKind.retextureModel(basicItemModel, woodType, textures)));
		}
	}

	private void retexturBlockModel(IRegistry<ModelResourceLocation, IBakedModel> registry,
			ImmutableMap<String, String> textures, IWoodType woodType, WoodBlockKind woodKind, IBlockState blockState,
			IWoodStateMapper woodMapper) {
		IModel basicModel = ModelLoaderRegistry
				.getModelOrMissing(woodMapper.getDefaultModelResourceLocation(blockState));
		if (basicModel instanceof MultipartModel) {
			MultipartModel multipartModel = (MultipartModel) basicModel;
			Multipart multipart = multipartModel.getMultipart();
			multipart.func_188138_a(blockState.func_177230_c().func_176194_O());
		}
		ModelResourceLocation basicLocation = woodMapper.getModelLocation(blockState);
		registry.func_82595_a(basicLocation,
				new SimpleRetexturedModel(woodKind.retextureModel(basicModel, woodType, textures)));

	}

	
	@Override
	public <T extends Block & IWoodTyped> void registerWoodModel(T woodTyped, boolean withVariants){
		woodModelEntrys.add(new WoodModelEntry(woodTyped, withVariants));
	}

	@Override
	public int getFoliageColorBasic() {
		return ColorizerFoliage.func_77468_c();
	}

	@Override
	public int getFoliageColorBirch() {
		return ColorizerFoliage.func_77469_b();
	}

	@Override
	public int getFoliageColorPine() {
		return ColorizerFoliage.func_77466_a();
	}
}
