/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.blocks;

import javax.annotation.Nonnull;
import java.util.EnumMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import forestry.core.tiles.TileUtil;
import net.minecraft.block.Block;
import net.minecraft.block.SoundType;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyBool;
import net.minecraft.block.properties.PropertyDirection;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.block.model.ModelResourceLocation;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumFacing.Plane;
import net.minecraft.util.EnumHand;
import net.minecraft.util.IStringSerializable;
import net.minecraft.util.Rotation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.TextComponentString;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

import net.minecraftforge.client.model.ModelLoader;
import net.minecraftforge.common.property.ExtendedBlockState;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.common.property.IUnlistedProperty;
import net.minecraftforge.common.property.Properties;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.core.ForestryAPI;
import forestry.api.core.ICamouflagedTile;
import forestry.api.core.IModelManager;
import forestry.api.core.ISpriteRegister;
import forestry.api.core.ITextureManager;
import forestry.api.core.Tabs;
import forestry.api.multiblock.IGreenhouseComponent;
import forestry.api.multiblock.IMultiblockController;
import forestry.core.CreativeTabForestry;
import forestry.core.blocks.BlockStructure;
import forestry.core.blocks.IBlockRotatable;
import forestry.core.blocks.IColoredBlock;
import forestry.core.blocks.propertys.UnlistedBlockAccess;
import forestry.core.blocks.propertys.UnlistedBlockPos;
import forestry.core.multiblock.MultiblockTileEntityForestry;
import forestry.core.tiles.IActivatable;
import forestry.core.utils.CamouflageUtil;
import forestry.core.utils.ItemStackUtil;
import forestry.core.utils.Log;
import forestry.greenhouse.tiles.TileGreenhouseClimateControl;
import forestry.greenhouse.tiles.TileGreenhouseControl;
import forestry.greenhouse.tiles.TileGreenhouseDoor;
import forestry.greenhouse.tiles.TileGreenhouseDryer;
import forestry.greenhouse.tiles.TileGreenhouseFan;
import forestry.greenhouse.tiles.TileGreenhouseGearbox;
import forestry.greenhouse.tiles.TileGreenhouseHatch;
import forestry.greenhouse.tiles.TileGreenhouseHeater;
import forestry.greenhouse.tiles.TileGreenhousePlain;
import forestry.greenhouse.tiles.TileGreenhouseSprinkler;
import forestry.greenhouse.tiles.TileGreenhouseValve;
import forestry.greenhouse.tiles.TileGreenhouseWindow;
import forestry.greenhouse.tiles.TileGreenhouseWindow.WindowMode;
import forestry.plugins.ForestryPluginUids;

public abstract class BlockGreenhouse extends BlockStructure implements ISpriteRegister, IColoredBlock, IBlockRotatable {
	private static final AxisAlignedBB SPRINKLER_BOUNDS = new AxisAlignedBB(0.3125F, 0.25F, 0.3125F, 0.6875F, 1F, 0.6875F);
	public static final PropertyEnum<State> STATE = PropertyEnum.func_177709_a("state", State.class);
	public static final PropertyBool BLOCKED = PropertyBool.func_177716_a("blocked");
	public static final PropertyDirection FACING = PropertyDirection.func_177712_a("facing", Plane.HORIZONTAL);
	
	public enum State implements IStringSerializable {
		ON, OFF;

		@Override
		public String func_176610_l() {
			return name().toLowerCase(Locale.ENGLISH);
		}
	}
	
	public static Map<BlockGreenhouseType, BlockGreenhouse> create() {
		Map<BlockGreenhouseType, BlockGreenhouse> blockMap = new EnumMap<>(BlockGreenhouseType.class);
		for (final BlockGreenhouseType type : BlockGreenhouseType.VALUES) {
			if (type == BlockGreenhouseType.BUTTERFLY_HATCH) {
				if (!ForestryAPI.enabledPlugins.contains(ForestryPluginUids.LEPIDOPTEROLOGY)) {
					continue;
				}
			}
			
			BlockGreenhouse block;
			if (type == BlockGreenhouseType.DOOR) {
				block = new BlockGreenhouseDoor();
			} else {
				block = new BlockGreenhouse() {
					@Nonnull
					@Override
					public BlockGreenhouseType getGreenhouseType() {
						return type;
					}
					
					@Nonnull
					@Override
					public SoundType func_185467_w() {
						if(type == BlockGreenhouseType.SPRINKLER || type == BlockGreenhouseType.GLASS || type == BlockGreenhouseType.WINDOW || type == BlockGreenhouseType.WINDOW_UP){
							return SoundType.field_185853_f;
						}
						return super.func_185467_w();
					}
				};
			}
			blockMap.put(type, block);
		}
		return blockMap;
	}
	
	public BlockGreenhouse() {
		super(Material.field_151576_e);
		BlockGreenhouseType greenhouseType = getGreenhouseType();
		IBlockState defaultState = this.field_176227_L.func_177621_b();
		if (greenhouseType.activatable && greenhouseType != BlockGreenhouseType.SPRINKLER) {
			defaultState = defaultState.func_177226_a(STATE, State.OFF);
		}
		if(greenhouseType == BlockGreenhouseType.WINDOW || greenhouseType == BlockGreenhouseType.WINDOW_UP){
			defaultState = defaultState.func_177226_a(FACING, EnumFacing.NORTH);
		}
		func_180632_j(defaultState);
		
		func_149711_c(1.0f);
		setHarvestLevel("pickaxe", 0);
		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.FARMING)) {
			func_149647_a(Tabs.tabAgriculture);
		} else {
			func_149647_a(CreativeTabForestry.tabForestry);
		}
	}
	
	@Override
	public boolean func_180639_a(World worldIn, BlockPos pos, IBlockState state, EntityPlayer player, EnumHand hand, ItemStack heldItem, EnumFacing side, float hitX, float hitY, float hitZ) {
		if (player.func_70093_af()) {
			return false;
		}

		TileEntity tile = worldIn.func_175625_s(pos);
		if (!(tile instanceof MultiblockTileEntityForestry)) {
			return false;
		}
		
		MultiblockTileEntityForestry part = (MultiblockTileEntityForestry) tile;
		IMultiblockController controller = part.getMultiblockLogic().getController();
		if(player.func_184614_ca() == null){
			if(tile instanceof TileGreenhouseWindow){
				TileGreenhouseWindow window = (TileGreenhouseWindow) tile;
				if(window.getMode() != WindowMode.CONTROL){
					if (!worldIn.field_72995_K) {
						if(window.isBlocked() == WindowMode.OPEN){
							if(window.getMode() == WindowMode.OPEN){
								window.setMode(WindowMode.PLAYER);
							}else{
								window.setMode(WindowMode.OPEN);	
							}
						}
					}
					worldIn.func_180498_a(player, getPlaySound(!window.isActive()), pos, 0);
					return true;
				}
			}
		
			if (player.func_184592_cb() == null) {
				// If the player's hands are empty and they right-click on a multiblock, they get a
				// multiblock-debugging message if the machine is not assembled.
				if (controller != null) {
					if (!controller.isAssembled()) {
						String validationError = controller.getLastValidationError();
						if (validationError != null) {
							long tick = worldIn.func_82737_E();
							if (tick > previousMessageTick + 20) {
								player.func_145747_a(new TextComponentString(validationError));
								previousMessageTick = tick;
							}
							return true;
						}
					}
				} else {
					player.func_145747_a(new TextComponentTranslation("for.multiblock.error.notConnected"));
					return true;
				}
			}
		}

		// Don't open the GUI if the multiblock isn't assembled
		if (controller == null || !controller.isAssembled()) {
			return false;
		}

		if (!worldIn.field_72995_K) {
			part.openGui(player);
		}
		return true;
	}
	
    protected int getPlaySound(boolean open){
        if (open){
        	return 1007;
        }else{
            return 1013;
        }
    }

	@Override
	@SideOnly(Side.CLIENT)
	public AxisAlignedBB func_180640_a(IBlockState state, World worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return SPRINKLER_BOUNDS.func_186670_a(pos);
		}
		return super.func_180640_a(state, worldIn, pos);
	}

	@Override
	public AxisAlignedBB func_180646_a(IBlockState blockState, World worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return SPRINKLER_BOUNDS;
		}
		return super.func_180646_a(blockState, worldIn, pos);
	}

	@Override
	public IBlockState getExtendedState(IBlockState state, IBlockAccess world, BlockPos pos) {
		if (getGreenhouseType() != BlockGreenhouseType.SPRINKLER && getGreenhouseType() != BlockGreenhouseType.WINDOW && getGreenhouseType() != BlockGreenhouseType.WINDOW_UP) {
			return ((IExtendedBlockState) super.getExtendedState(state, world, pos)).withProperty(UnlistedBlockPos.POS, pos)
					.withProperty(UnlistedBlockAccess.BLOCKACCESS, world);
		}
		return super.getExtendedState(state, world, pos);
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return new ExtendedBlockState(this, new IProperty[]{Properties.StaticProperty}, new IUnlistedProperty[]{Properties.AnimationProperty});
		}else if (getGreenhouseType() == BlockGreenhouseType.WINDOW || getGreenhouseType() == BlockGreenhouseType.WINDOW_UP) {
			return new BlockStateContainer(this, STATE, FACING);
		}  else if (getGreenhouseType().activatable) {
			return new ExtendedBlockState(this, new IProperty[]{STATE}, new IUnlistedProperty[]{UnlistedBlockPos.POS, UnlistedBlockAccess.BLOCKACCESS});
		} else {
			return new ExtendedBlockState(this, new IProperty[0], new IUnlistedProperty[]{UnlistedBlockPos.POS, UnlistedBlockAccess.BLOCKACCESS});
		}
	}
	
	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return state.func_177226_a(Properties.StaticProperty, true);
		} else {
			TileEntity tile = TileUtil.getTile(worldIn, pos, TileEntity.class);
			if(tile instanceof IActivatable){
				state = state.func_177226_a(STATE, ((IActivatable) tile).isActive() ? State.ON : State.OFF);
			}
			if(tile instanceof TileGreenhouseWindow){
				TileGreenhouseWindow window = (TileGreenhouseWindow) tile;
			}
			return super.func_176221_a(state, worldIn, pos);
		}
	}
	
	@Override
	public IBlockState func_185499_a(IBlockState state, Rotation rot) {
		EnumFacing facing = state.func_177229_b(FACING);
		return state.func_177226_a(FACING, rot.func_185831_a(facing));
	}
	
	@Override
	public IBlockState func_176203_a(int meta) {
		if(getGreenhouseType() == BlockGreenhouseType.WINDOW || getGreenhouseType() == BlockGreenhouseType.WINDOW_UP){
			return func_176223_P().func_177226_a(FACING, EnumFacing.field_82609_l[meta + 2]);
		}
		return super.func_176203_a(meta);
	}
	
	@Override
	public int func_176201_c(IBlockState state) {
		if(getGreenhouseType() == BlockGreenhouseType.WINDOW || getGreenhouseType() == BlockGreenhouseType.WINDOW_UP){
			return state.func_177229_b(FACING).ordinal() - 2;
		}
		return 0;
	}
	
	@Override
	public void rotateAfterPlacement(EntityPlayer player, World world, BlockPos pos, EnumFacing side) {
		IBlockState state = world.func_180495_p(pos);

		if(state.func_177228_b().containsKey(FACING)){
			world.func_175656_a(pos, state.func_177226_a(FACING, player.func_174811_aO().func_176734_d()));
		}
	}
	
	@Override
	public void func_189540_a(IBlockState state, World worldIn, BlockPos pos, Block blockIn) {
		try {
			TileEntity tile = worldIn.func_175625_s(pos);
			if (tile instanceof TileGreenhouseWindow) {
				((TileGreenhouseWindow) tile).onNeighborBlockChange();
			}
		} catch (StackOverflowError error) {
			Log.error("Stack Overflow Error in BlockMachine.onNeighborBlockChange()", error);
			throw error;
		}
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public void func_149666_a(Item item, CreativeTabs tab, List<ItemStack> list) {
		list.add(new ItemStack(item));
	}

	@Override
	public TileEntity func_149915_a(World world, int meta) {
		BlockGreenhouseType type = getGreenhouseType();
		switch (type) {
			case GEARBOX:
				return new TileGreenhouseGearbox();
			case SPRINKLER:
				return new TileGreenhouseSprinkler();
			case DRYER:
				return new TileGreenhouseDryer();
			case VALVE:
				return new TileGreenhouseValve();
			case FAN:
				return new TileGreenhouseFan();
			case HEATER:
				return new TileGreenhouseHeater();
			case DOOR:
				return new TileGreenhouseDoor();
			case CONTROL:
				return new TileGreenhouseControl();
			case CLIMATE_CONTROL:
				return new TileGreenhouseClimateControl();
			case WINDOW_UP:
			case WINDOW:
				return new TileGreenhouseWindow();
			case HATCH_INPUT:
			case HATCH_OUTPUT:
				return new TileGreenhouseHatch();
			case BUTTERFLY_HATCH:
				return new TileGreenhouseHatch();
			default:
				return new TileGreenhousePlain();
		}
	}

	@Override
	public int colorMultiplier(IBlockState state, IBlockAccess worldIn, BlockPos pos, int tintIndex) {
		if(pos == null){
			return 0xffffff;
		}
		TileEntity tile = worldIn.func_175625_s(pos);
		if (tile instanceof ICamouflagedTile) {
			ItemStack camouflageStack = CamouflageUtil.getCamouflageBlock(worldIn, pos);

			if (tintIndex < 100 && camouflageStack != null) {
				Block block = Block.func_149634_a(camouflageStack.func_77973_b());
				if(block != null){
					IBlockState camouflageState = block.func_176203_a(camouflageStack.func_77952_i());
					
					int color = Minecraft.func_71410_x().func_184125_al().func_186724_a(camouflageState, worldIn, pos, tintIndex);
					if(color != -1){
						return color;
					}
				}
			}
		}

		return 0xffffff;
	}

	/* MODELS */
	@Override
	@SideOnly(Side.CLIENT)
	public BlockRenderLayer func_180664_k() {
		if (getGreenhouseType() == BlockGreenhouseType.GLASS || getGreenhouseType() == BlockGreenhouseType.WINDOW || getGreenhouseType() == BlockGreenhouseType.WINDOW_UP || getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			return BlockRenderLayer.TRANSLUCENT;
		}
		return BlockRenderLayer.CUTOUT;
	}

	@Override
	public boolean func_149686_d(IBlockState state) {
		return getGreenhouseType() != BlockGreenhouseType.GLASS && getGreenhouseType() != BlockGreenhouseType.SPRINKLER && getGreenhouseType() != BlockGreenhouseType.WINDOW && getGreenhouseType() != BlockGreenhouseType.WINDOW_UP;
	}

	@Override
	public boolean func_149662_c(IBlockState state) {
		return getGreenhouseType() != BlockGreenhouseType.GLASS && getGreenhouseType() != BlockGreenhouseType.SPRINKLER && getGreenhouseType() != BlockGreenhouseType.WINDOW && getGreenhouseType() != BlockGreenhouseType.WINDOW_UP;
	}
	
	@Override
	public boolean func_176196_c(World worldIn, BlockPos pos) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			if (worldIn.func_180495_p(pos.func_177984_a()).func_177230_c() == this) {
				return false;
			}
			if (!(worldIn.func_175625_s(pos.func_177984_a()) instanceof IGreenhouseComponent)) {
				return false;
			}
		}
		return super.func_176196_c(worldIn, pos);
	}
	
	@Override
	public void func_180633_a(World world, BlockPos pos, IBlockState state, EntityLivingBase placer, ItemStack stack) {
		super.func_180633_a(world, pos, state, placer, stack);
		TileEntity tile = world.func_175625_s(pos);
		if(tile instanceof TileGreenhouseWindow){
			TileGreenhouseWindow window = (TileGreenhouseWindow) tile;
			if(!window.func_145831_w().field_72995_K){
				window.setMode(window.isBlocked());
			}
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public boolean func_176225_a(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		IBlockState iblockstate = blockAccess.func_180495_p(pos);
		Block block = iblockstate.func_177230_c();

		if (getGreenhouseType() == BlockGreenhouseType.GLASS) {
			BlockPos posSide = pos.func_177972_a(side);
			if (blockAccess.func_180495_p(posSide) != iblockstate) {
				return true;
			}
			TileEntity tile = blockAccess.func_175625_s(pos);
			TileEntity tileSide = blockAccess.func_175625_s(posSide);
			if(tile instanceof TileGreenhousePlain && tileSide instanceof TileGreenhousePlain){
				if(((TileGreenhousePlain)tile).getCamouflageType().equals(((TileGreenhousePlain)tileSide).getCamouflageType())){
					ItemStack camouflage = CamouflageUtil.getCamouflageBlock(blockAccess, pos);
					ItemStack camouflageSide = CamouflageUtil.getCamouflageBlock(blockAccess, posSide);
					if(camouflage != null && camouflageSide != null){
						if(ItemStackUtil.isIdenticalItem(camouflage, camouflageSide)){
							return false;
						}
						return true;
					}
				}
			}
			return block != this && super.func_176225_a(blockState, blockAccess, pos, side);
		} else if (getGreenhouseType() == BlockGreenhouseType.DOOR) {
			return super.func_176225_a(blockState, blockAccess, pos, side);
		}

		return super.func_176225_a(blockState, blockAccess, pos, side);
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public void registerModel(Item item, IModelManager manager) {
		if (getGreenhouseType() == BlockGreenhouseType.SPRINKLER) {
			ModelLoader.setCustomModelResourceLocation(item, 0, new ModelResourceLocation("forestry:greenhouse.sprinkler", "inventory"));
		} else if (getGreenhouseType() == BlockGreenhouseType.WINDOW) {
			ModelLoader.setCustomModelResourceLocation(item, 0, new ModelResourceLocation("forestry:greenhouse.window", "inventory"));
		} else if (getGreenhouseType() == BlockGreenhouseType.WINDOW_UP) {
			ModelLoader.setCustomModelResourceLocation(item, 0, new ModelResourceLocation("forestry:greenhouse.window_up", "inventory"));
		}  else {
			ModelLoader.setCustomModelResourceLocation(item, 0, new ModelResourceLocation("forestry:greenhouse", "inventory"));
		}
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public void registerSprites(ITextureManager manager) {
		BlockGreenhouseType.registerSprites();
	}

	@Override
	public boolean canConnectRedstone(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side) {
		return getGreenhouseType() == BlockGreenhouseType.CONTROL;
	}
	
	@Nonnull
	public abstract BlockGreenhouseType getGreenhouseType();
}
