package forestry.energy;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;

import forestry.api.core.INbtReadable;
import forestry.api.core.INbtWritable;
import forestry.core.network.DataInputStreamForestry;
import forestry.core.network.DataOutputStreamForestry;
import forestry.core.network.IStreamable;
import forestry.energy.compat.EnergyStorageWrapper;
import forestry.energy.compat.tesla.TeslaConsumerWrapper;
import forestry.energy.compat.tesla.TeslaHelper;
import forestry.energy.compat.tesla.TeslaHolderWrapper;
import forestry.energy.compat.tesla.TeslaProducerWrapper;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.energy.CapabilityEnergy;
import net.minecraftforge.energy.EnergyStorage;
import net.minecraftforge.energy.IEnergyStorage;

public class EnergyManager extends EnergyStorage implements IStreamable, INbtReadable, INbtWritable {
	@Nonnull
	private EnergyTransferMode externalMode = EnergyTransferMode.BOTH;

	public EnergyManager(int maxTransfer, int capacity) {
		super(EnergyHelper.scaleForDifficulty(capacity), EnergyHelper.scaleForDifficulty(maxTransfer), EnergyHelper.scaleForDifficulty(maxTransfer));
	}

	public void setExternalMode(@Nonnull EnergyTransferMode externalMode) {
		this.externalMode = externalMode;
	}

	@Nonnull
	public EnergyTransferMode getExternalMode() {
		return externalMode;
	}

	@Override
	public void readFromNBT(NBTTagCompound nbt) {
		final int energy;
		if (nbt.func_74764_b("EnergyManager")) { // legacy
			NBTTagCompound energyManagerNBT = nbt.func_74775_l("EnergyManager");
			NBTTagCompound energyStorageNBT = energyManagerNBT.func_74775_l("EnergyStorage");
			energy = energyStorageNBT.func_74762_e("Energy");
		} else {
			energy = nbt.func_74762_e("Energy");
		}

		setEnergyStored(energy);
	}

	@Override
	public NBTTagCompound writeToNBT(NBTTagCompound nbt) {
		nbt.func_74768_a("Energy", energy);
		return nbt;
	}

	@Override
	public void writeData(DataOutputStreamForestry data) throws IOException {
		data.writeVarInt(this.energy);
	}

	@Override
	public void readData(DataInputStreamForestry data) throws IOException {
		int energyStored = data.readVarInt();
		setEnergyStored(energyStored);
	}

	public int getMaxEnergyReceived() {
		return this.maxReceive;
	}

	/**
	 * Drains an amount of energy, due to decay from lack of work or other factors
	 */
	public void drainEnergy(int amount) {
		setEnergyStored(energy - amount);
	}

	/**
	 * Creates an amount of energy, generated by engines
	 */
	public void generateEnergy(int amount) {
		setEnergyStored(energy + amount);
	}

	public void setEnergyStored(int energyStored) {
		this.energy = energyStored;
		if (this.energy > capacity) {
			this.energy = capacity;
		} else if (this.energy < 0) {
			this.energy = 0;
		}
	}

	public boolean hasCapability(Capability<?> capability) {
		if (capability == null) {
			return false;
		}

		if (capability == CapabilityEnergy.ENERGY) {
			return true;
		} else if (capability == TeslaHelper.TESLA_PRODUCER && externalMode.canExtract()) {
			return true;
		} else if (capability == TeslaHelper.TESLA_CONSUMER && externalMode.canReceive()) {
			return true;
		} else if (capability == TeslaHelper.TESLA_HOLDER) {
			return true;
		} else {
			return false;
		}
	}

	@Nullable
	public <T> T getCapability(Capability<T> capability) {
		if (capability == null) {
			return null;
		}

		if (capability == CapabilityEnergy.ENERGY) {
			IEnergyStorage energyStorage = new EnergyStorageWrapper(this, externalMode);
			return CapabilityEnergy.ENERGY.cast(energyStorage);
		} else if (capability == TeslaHelper.TESLA_PRODUCER && externalMode.canExtract()) {
			return TeslaHelper.TESLA_PRODUCER.cast(new TeslaProducerWrapper(this));
		} else if (capability == TeslaHelper.TESLA_CONSUMER && externalMode.canReceive()) {
			return TeslaHelper.TESLA_CONSUMER.cast(new TeslaConsumerWrapper(this));
		} else if (capability == TeslaHelper.TESLA_HOLDER) {
			return TeslaHelper.TESLA_HOLDER.cast(new TeslaHolderWrapper(this));
		} else {
			return null;
		}
	}

}
