package forestry.arboriculture.models;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Function;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableMap;
import forestry.api.arboriculture.IAlleleTreeSpecies;
import forestry.api.arboriculture.TreeManager;
import forestry.arboriculture.blocks.BlockPile;
import forestry.arboriculture.genetics.TreeDefinition;
import forestry.arboriculture.tiles.TilePile;
import forestry.core.blocks.propertys.UnlistedBlockAccess;
import forestry.core.blocks.propertys.UnlistedBlockPos;
import forestry.core.models.BlankModel;
import forestry.core.models.DefaultTextureGetter;
import forestry.core.tiles.TileUtil;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.block.model.BakedQuad;
import net.minecraft.client.renderer.block.model.IBakedModel;
import net.minecraft.client.renderer.block.model.ItemOverrideList;
import net.minecraft.client.renderer.block.model.ModelRotation;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.client.event.ModelBakeEvent;
import net.minecraftforge.client.model.IModel;
import net.minecraftforge.client.model.ModelLoaderRegistry;
import net.minecraftforge.client.model.ModelProcessingHelper;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

@SideOnly(Side.CLIENT)
public class ModelWoodPile extends BlankModel{

	private final Function<ResourceLocation, TextureAtlasSprite> textureGetter = new DefaultTextureGetter();
	private static IModel modelWoodPileItem;
	private static IModel modelWoodPileBlock;
	private static final Cache<String, IBakedModel> blockCache = CacheBuilder.newBuilder().expireAfterAccess(1, TimeUnit.MINUTES).build();
	private static final Cache<String, IBakedModel> itemCache = CacheBuilder.newBuilder().expireAfterAccess(1, TimeUnit.MINUTES).build();
	
	/**
	 * Init the model with datas from the ModelBakeEvent.
	 */
	public static void onModelBake(ModelBakeEvent event){
		blockCache.invalidateAll();
		itemCache.invalidateAll();
	}
	
	@Override
	public List<BakedQuad> func_188616_a(IBlockState state, EnumFacing side, long rand) {
		if(state instanceof IExtendedBlockState){
			IExtendedBlockState stateExtended = (IExtendedBlockState) state;
	
			IBlockAccess world = stateExtended.getValue(UnlistedBlockAccess.BLOCKACCESS);
			BlockPos pos = stateExtended.getValue(UnlistedBlockPos.POS);
			
			TilePile pile = TileUtil.getTile(world, pos, TilePile.class);
			
			if(pile == null){
				return Collections.emptyList();
			}
			IAlleleTreeSpecies treeSpecies = pile.getTreeSpecies();
			if (treeSpecies == null) {
				treeSpecies = (IAlleleTreeSpecies) TreeManager.treeRoot.getDefaultTemplate()[TreeManager.treeRoot.getSpeciesChromosomeType().ordinal()];
			}
			
			if (modelWoodPileBlock == null) {
				try {
					modelWoodPileBlock = ModelLoaderRegistry.getModel(new ResourceLocation("forestry:block/woodPile"));
				} catch (Exception e) {
					return Collections.emptyList();
				}
				if (modelWoodPileBlock == null) {
					return Collections.emptyList();
				}
			}

			IBakedModel model = bakeModel(treeSpecies, false);
			return model.func_188616_a(state, side, rand);
		}
		return Collections.emptyList();
	}

	@Override
	public boolean func_177555_b() {
		return false;
	}

	@Override
	public boolean func_177556_c() {
		return true;
	}

	@Override
	public TextureAtlasSprite func_177554_e() {
		return TreeDefinition.Oak.getGenome().getPrimary().getWoodProvider().getSprite(false);
	}
	
	@Override
	protected ItemOverrideList createOverrides() {
		return new PileItemOverrideList();
	}

	private IBakedModel bakeModel(IAlleleTreeSpecies treeSpecies, boolean isItem) {
		if(treeSpecies == null || treeSpecies.getWoodProvider() == null || treeSpecies.getWoodProvider().getSprite(false) == null){
			return null;
		}
		ImmutableMap.Builder<String, String> textures = ImmutableMap.builder();
		String treeUID = treeSpecies.getUID();
		Cache<String, IBakedModel> map = isItem ? itemCache : blockCache;
		if(map.getIfPresent(treeUID) == null){
			textures.put("woodBark", treeSpecies.getWoodProvider().getSprite(false).func_94215_i());
			textures.put("woodTop", treeSpecies.getWoodProvider().getSprite(true).func_94215_i());
			IModel retextureWoodPile = ModelProcessingHelper.retexture(isItem ? modelWoodPileItem : modelWoodPileBlock, textures.build());
			map.put(treeUID, retextureWoodPile.bake(ModelRotation.X0_Y0, isItem ? DefaultVertexFormats.field_176599_b : DefaultVertexFormats.field_176600_a, textureGetter));
		}
		return map.getIfPresent(treeUID);
	}
	
	private class PileItemOverrideList extends ItemOverrideList {
		public PileItemOverrideList() {
			super(Collections.emptyList());
		}

		@Override
		public IBakedModel handleItemState(IBakedModel originalModel, ItemStack stack, World world, EntityLivingBase entity) {
			if (modelWoodPileItem == null) {
				try {
					modelWoodPileItem = ModelLoaderRegistry.getModel(new ResourceLocation("forestry:item/woodPile"));
				} catch (Exception e) {
					return null;
				}
				if (modelWoodPileItem == null) {
					return null;
				}
			}
			IAlleleTreeSpecies treeSpecies = BlockPile.getTreeSpecies(stack);
			if (treeSpecies == null) {
				treeSpecies = (IAlleleTreeSpecies) TreeManager.treeRoot.getDefaultTemplate()[TreeManager.treeRoot.getSpeciesChromosomeType().ordinal()];
			}
			return bakeModel(treeSpecies, true);
		}
	}

}
