/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.lepidopterology.entities;

import net.minecraft.block.Block;
import net.minecraft.block.BlockFence;
import net.minecraft.block.BlockFlower;
import net.minecraft.block.BlockWall;
import net.minecraft.block.IGrowable;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityCreature;
import net.minecraft.entity.IEntityLivingData;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.datasync.DataParameter;
import net.minecraft.network.datasync.DataSerializers;
import net.minecraft.network.datasync.EntityDataManager;
import net.minecraft.util.DamageSource;
import net.minecraft.util.EnumHand;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.DifficultyInstance;
import net.minecraft.world.World;
import net.minecraft.world.chunk.Chunk;

import net.minecraftforge.common.IPlantable;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.arboriculture.EnumGermlingType;
import forestry.api.arboriculture.TreeManager;
import forestry.api.core.IToolScoop;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IAllele;
import forestry.api.genetics.IIndividual;
import forestry.api.genetics.ISpeciesRoot;
import forestry.api.lepidopterology.ButterflyManager;
import forestry.api.lepidopterology.EnumFlutterType;
import forestry.api.lepidopterology.IAlleleButterflySpecies;
import forestry.api.lepidopterology.IButterfly;
import forestry.api.lepidopterology.IButterflyGenome;
import forestry.api.lepidopterology.IButterflyRoot;
import forestry.api.lepidopterology.IEntityButterfly;
import forestry.api.lepidopterology.ILepidopteristTracker;
import forestry.core.utils.ItemStackUtil;
import forestry.lepidopterology.genetics.Butterfly;

public class EntityButterfly extends EntityCreature implements IEntityButterfly {

	/* CONSTANTS */
	public static final int COOLDOWNS = 1500;

	private static final DataParameter<String> DATAWATCHER_ID_SPECIES = EntityDataManager.func_187226_a(EntityButterfly.class, DataSerializers.field_187194_d);
	private static final DataParameter<Integer> DATAWATCHER_ID_SIZE = EntityDataManager.func_187226_a(EntityButterfly.class, DataSerializers.field_187192_b);
	private static final DataParameter<Byte> DATAWATCHER_ID_STATE = EntityDataManager.func_187226_a(EntityButterfly.class, DataSerializers.field_187191_a);

	private static final float DEFAULT_BUTTERFLY_SIZE = 0.75f;
	private static final EnumButterflyState DEFAULT_STATE = EnumButterflyState.FLYING;

	public static final int EXHAUSTION_REST = 1000;
	public static final int EXHAUSTION_CONSUMPTION = 100 * EXHAUSTION_REST;
	public static final int MAX_LIFESPAN = 24000 * 7; // one minecraft week in ticks

	private Vec3d flightTarget;
	private int exhaustion;
	private IButterfly contained;
	private IIndividual pollen;

	public int cooldownPollination = 0;
	public int cooldownEgg = 0;
	public int cooldownMate = 0;

	// Client Rendering
	private IAlleleButterflySpecies species;
	private float size = DEFAULT_BUTTERFLY_SIZE;
	private EnumButterflyState state = DEFAULT_STATE;

	/* CONSTRUCTOR */
	public EntityButterfly(World world) {
		super(world);
		setDefaults();
	}

	public EntityButterfly(World world, IButterfly butterfly) {
		super(world);
		setDefaults();
		setIndividual(butterfly);
	}

	@Override
	protected void func_70088_a() {
		super.func_70088_a();

		field_70180_af.func_187214_a(DATAWATCHER_ID_SPECIES, "");
		field_70180_af.func_187214_a(DATAWATCHER_ID_SIZE, (int) (DEFAULT_BUTTERFLY_SIZE * 100));
		field_70180_af.func_187214_a(DATAWATCHER_ID_STATE, (byte) DEFAULT_STATE.ordinal());
	}

	private void setDefaults() {
		field_70714_bg.func_75776_a(8, new AIButterflyFlee(this));
		field_70714_bg.func_75776_a(9, new AIButterflyMate(this));
		field_70714_bg.func_75776_a(10, new AIButterflyPollinate(this));
		field_70714_bg.func_75776_a(11, new AIButterflyRest(this));
		field_70714_bg.func_75776_a(12, new AIButterflyRise(this));
		field_70714_bg.func_75776_a(12, new AIButterflyWander(this));
		field_70714_bg.func_75776_a(13, new AIButterflyGoHome(this));
	}

	@Override
	public EntityCreature getEntity() {
		return this;
	}

	/* SAVING & LOADING */
	@Override
	public void func_70014_b(NBTTagCompound nbttagcompound) {
		super.func_70014_b(nbttagcompound);

		if (contained != null) {
			NBTTagCompound bio = new NBTTagCompound();
			contained.writeToNBT(bio);
			nbttagcompound.func_74782_a("BTFLY", bio);
		}

		if (pollen != null) {
			NBTTagCompound pln = new NBTTagCompound();
			pollen.writeToNBT(pln);
			nbttagcompound.func_74782_a("PLN", pln);
		}

		nbttagcompound.func_74774_a("STATE", (byte) getState().ordinal());
		nbttagcompound.func_74768_a("EXH", exhaustion);
	}

	@Override
	public void func_70037_a(NBTTagCompound nbttagcompound) {
		super.func_70037_a(nbttagcompound);

		IButterfly butterfly = null;
		if (nbttagcompound.func_74764_b("BTFLY")) {
			butterfly = new Butterfly((NBTTagCompound) nbttagcompound.func_74781_a("BTFLY"));
		}
		setIndividual(butterfly);

		if (nbttagcompound.func_74764_b("PLN")) {
			pollen = TreeManager.treeRoot.getMember((NBTTagCompound) nbttagcompound.func_74781_a("PLN"));
		}

		EnumButterflyState state = EnumButterflyState.VALUES[nbttagcompound.func_74771_c("STATE")];
		setState(state);
		exhaustion = nbttagcompound.func_74762_e("EXH");
	}

	public float getWingFlap(float partialTicktime) {
		return getState().getWingFlap(this, species.getUID().hashCode(), partialTicktime);
	}

	/* STATE - Used for AI and rendering */
	public void setState(EnumButterflyState state) {
		if (this.state != state) {
			this.state = state;
			if (!field_70170_p.field_72995_K) {
				field_70180_af.func_187227_b(DATAWATCHER_ID_STATE, (byte) state.ordinal());
			}
		}
	}

	public EnumButterflyState getState() {
		return state;
	}

	public float getSize() {
		return size;
	}

	public float getSpeed() {
		return contained.getGenome().getSpeed();
	}

	/* DESTINATION */
	public Vec3d getDestination() {
		return flightTarget;
	}

	public void setDestination(Vec3d destination) {
		flightTarget = destination;
	}

	@Override
	public float func_180484_a(BlockPos pos) {
		if (!field_70170_p.func_175667_e(pos)) {
			return -100f;
		}

		float weight = 0.0f;

		if (!getButterfly().isAcceptedEnvironment(field_70170_p, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p())) {
			weight -= 15.0f;
		}

		if (!field_70170_p.func_72872_a(EntityButterfly.class, new AxisAlignedBB(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p(), pos.func_177958_n() + 1, pos.func_177956_o() + 1, pos.func_177952_p() + 1)).isEmpty()) {
			weight -= 1.0f;
		}

		int depth = getFluidDepth(pos);
		if (depth > 0) {
			weight -= 0.1f * depth;
		} else {
			IBlockState blockState = field_70170_p.func_180495_p(pos);
			Block block = blockState.func_177230_c();
			if (block instanceof BlockFlower) {
				weight += 2.0f;
			} else if (block instanceof IPlantable) {
				weight += 1.5f;
			} else if (block instanceof IGrowable) {
				weight += 1.0f;
			} else if (block.func_149688_o(blockState) == Material.field_151585_k) {
				weight += 1.0f;
			}

			BlockPos posBelow = pos.func_177977_b();
			IBlockState blockStateBelow = field_70170_p.func_180495_p(posBelow);
			Block blockBelow = blockStateBelow.func_177230_c();
			if (blockBelow.isLeaves(blockStateBelow, field_70170_p, posBelow)) {
				weight += 2.5f;
			} else if (blockBelow instanceof BlockFence) {
				weight += 1.0f;
			} else if (blockBelow instanceof BlockWall) {
				weight += 1.0f;
			}
		}

		weight += field_70170_p.func_175724_o(pos);
		return weight;
	}

	private int getFluidDepth(BlockPos pos) {
		Chunk chunk = field_70170_p.func_175726_f(pos);
		int xx = pos.func_177958_n() & 15;
		int zz = pos.func_177952_p() & 15;
		int depth = 0;
		for (int y = chunk.func_76625_h() + 15; y > 0; --y) {
			IBlockState blockState = chunk.func_186032_a(xx, y, zz);
			Block block = blockState.func_177230_c();
			if (block.func_149688_o(blockState).func_76224_d()) {
				depth++;
			} else if (!block.isAir(blockState, field_70170_p, pos)) {
				break;
			}
		}

		return depth;
	}

	/* POLLEN */
	@Override
	public IIndividual getPollen() {
		return pollen;
	}

	@Override
	public void setPollen(IIndividual pollen) {
		this.pollen = pollen;
	}

	public boolean isRenderable() {
		return species != null;
	}

	/* EXHAUSTION */
	@Override
	public void changeExhaustion(int change) {
		exhaustion = exhaustion + change > 0 ? exhaustion + change : 0;
	}

	@Override
	public int getExhaustion() {
		return exhaustion;
	}

	/* FLYING ABILITY */
	public boolean canFly() {
		return contained.canTakeFlight(field_70170_p, field_70165_t, field_70163_u, field_70161_v);
	}

	public void setIndividual(IButterfly butterfly) {
		if (butterfly == null) {
			butterfly = ButterflyManager.butterflyRoot.templateAsIndividual(ButterflyManager.butterflyRoot.getDefaultTemplate());
		}
		contained = butterfly;

		IButterflyGenome genome = contained.getGenome();

		field_70178_ae = genome.getFireResist();
		size = genome.getSize();
		func_70105_a(size, 0.4f);
		species = genome.getPrimary();

		if (!field_70170_p.field_72995_K) {
			field_70180_af.func_187227_b(DATAWATCHER_ID_SIZE, (int) (size * 100));
			field_70180_af.func_187227_b(DATAWATCHER_ID_SPECIES, species.getUID());
		} else {
			textureResource = new ResourceLocation(species.getEntityTexture());
		}
	}

	@Override
	public IButterfly getButterfly() {
		return contained;
	}
	
	@Override
	public IEntityLivingData func_180482_a(DifficultyInstance difficulty, IEntityLivingData data) {
		if (!field_70170_p.field_72995_K) {
			setIndividual(contained);
		}
		return data;
	}

	@Override
	public String func_70005_c_() {
		if (species == null) {
			return super.func_70005_c_();
		}
		return species.getName();
	}

	@Override
	public boolean func_70601_bi() {
		return true;
	}

	@Override
	public int func_82145_z() {
		return 1000;
	}

	@SideOnly(Side.CLIENT)
	private ResourceLocation textureResource;

	@SideOnly(Side.CLIENT)
	public ResourceLocation getTexture() {
		return textureResource;
	}

	@Override
	public boolean func_70104_M() {
		return false;
	}

	@Override
	protected void func_82167_n(Entity other) {
	}

	@Override
	public boolean func_175446_cd() {
		return false;
	}

	@Override
	protected boolean func_70692_ba() {
		return field_70173_aa > MAX_LIFESPAN;
	}

	/* INTERACTION */

	@Override
	protected boolean func_184645_a(EntityPlayer player, EnumHand hand, ItemStack stack) {
		if (field_70128_L) {
			return false;
		}

		if (stack == null) {
			return false;
		}

		if (!(stack.func_77973_b() instanceof IToolScoop)) {
			return false;
		}

		if (!field_70170_p.field_72995_K) {
			IButterflyRoot root = contained.getGenome().getPrimary().getRoot();
			ILepidopteristTracker tracker = root.getBreedingTracker(field_70170_p, player.func_146103_bH());
			ItemStack itemStack = root.getMemberStack(contained.copy(), EnumFlutterType.BUTTERFLY);

			tracker.registerCatch(contained);
			ItemStackUtil.dropItemStackAsEntity(itemStack, field_70170_p, field_70165_t, field_70163_u, field_70161_v);
			func_70106_y();
		} else {
			player.func_184609_a(hand);
		}
		return true;
	}

	/* LOOT */
	@Override
	protected void func_70628_a(boolean playerKill, int lootLevel) {
		for (ItemStack stack : contained.getLootDrop(this, playerKill, lootLevel)) {
			ItemStackUtil.dropItemStackAsEntity(stack, field_70170_p, field_70165_t, field_70163_u, field_70161_v);
		}

		// Drop pollen if any
		IIndividual pollen = getPollen();
		if (pollen != null) {
			ISpeciesRoot root = AlleleManager.alleleRegistry.getSpeciesRoot(pollen.getClass());
			ItemStack pollenStack = root.getMemberStack(pollen, EnumGermlingType.POLLEN);
			ItemStackUtil.dropItemStackAsEntity(pollenStack, field_70170_p, field_70165_t, field_70163_u, field_70161_v);
		}
	}

	/* UPDATING */
	@Override
	public void func_70071_h_() {
		super.func_70071_h_();

		// Update stuff client side
		if (field_70170_p.field_72995_K) {
			if (species == null) {
				String speciesUid = field_70180_af.func_187225_a(DATAWATCHER_ID_SPECIES);
				IAllele allele = AlleleManager.alleleRegistry.getAllele(speciesUid);
				if (allele instanceof IAlleleButterflySpecies) {
					species = (IAlleleButterflySpecies) allele;
					textureResource = new ResourceLocation(species.getEntityTexture());
					size = field_70180_af.func_187225_a(DATAWATCHER_ID_SIZE) / 100f;
				}
			}

			byte stateOrdinal = field_70180_af.func_187225_a(DATAWATCHER_ID_STATE);
			if (state == null || state.ordinal() != stateOrdinal) {
				setState(EnumButterflyState.VALUES[stateOrdinal]);
			}
		}

		field_70181_x *= 0.6000000238418579d;

		// Make sure we die if the butterfly hasn't rested in a long, long time.
		if (exhaustion > EXHAUSTION_CONSUMPTION && func_70681_au().nextInt(20) == 0) {
			func_70097_a(DamageSource.field_76377_j, 1);
		}

		if (field_70173_aa > MAX_LIFESPAN) {
			func_70097_a(DamageSource.field_76377_j, 1);
		}

		// Reduce cooldowns
		if (cooldownEgg > 0) {
			cooldownEgg--;
		}
		if (cooldownPollination > 0) {
			cooldownPollination--;
		}
		if (cooldownMate > 0) {
			cooldownMate--;
		}
	}

	@Override
	protected void func_70619_bc() {
		super.func_70619_bc();

		if (getState().doesMovement && flightTarget != null) {
			double diffX = flightTarget.field_72450_a + 0.5d - field_70165_t;
			double diffY = flightTarget.field_72448_b + 0.1d - field_70163_u;
			double diffZ = flightTarget.field_72449_c + 0.5d - field_70161_v;

			field_70159_w += (Math.signum(diffX) * 0.5d - field_70159_w) * 0.10000000149011612d;
			field_70181_x += (Math.signum(diffY) * 0.699999988079071d - field_70181_x) * 0.10000000149011612d;
			field_70179_y += (Math.signum(diffZ) * 0.5d - field_70179_y) * 0.10000000149011612d;

			float horizontal = (float) (Math.atan2(field_70179_y, field_70159_w) * 180d / Math.PI) - 90f;
			field_70177_z += MathHelper.func_76142_g(horizontal - field_70177_z);

			func_70657_f(contained.getGenome().getSpeed());
		}
	}

	@Override
	protected boolean func_70041_e_() {
		return false;
	}

	@Override
	public void func_180430_e(float distance, float damageMultiplier) {
	}

	@Override
	protected void func_184231_a(double y, boolean onGroundIn, IBlockState state, BlockPos pos) {
	}

	@Override
	public boolean func_145773_az() {
		return true;
	}

	@Override
	public float func_70678_g(float partialTicktime) {
		float flap = field_70733_aJ - field_70732_aI;
		if (flap < 0.0F) {
			++flap;
		}

		return field_70732_aI + flap * partialTicktime;

	}

	@Override
	protected float func_70599_aP() {
		return 0.1F;
	}

	@Override
	public ItemStack getPickedResult(RayTraceResult target) {
		if (species == null) {
			return null;
		}
		IButterflyRoot root = species.getRoot();
		IAllele[] template = root.getTemplate(species.getUID());
		IButterfly butterfly = root.templateAsIndividual(template);
		return root.getMemberStack(butterfly, EnumFlutterType.BUTTERFLY);
	}

	@Override
	public boolean canMateWith(IEntityButterfly butterfly) {
		if(butterfly == null || butterfly.getButterfly() == null || butterfly.getButterfly().getMate() != null){
			return false;
		}
		if (getButterfly() == null || getButterfly().getMate() != null) {
			return false;
		}
		return !getButterfly().isGeneticEqual(butterfly.getButterfly());
	}

	@Override
	public boolean canMate() {
		return cooldownMate <= 0;
	}
}
