/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.farming.logic;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import net.minecraft.block.Block;
import net.minecraft.block.BlockOldLog;
import net.minecraft.block.BlockPlanks;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import forestry.api.farming.FarmDirection;
import forestry.api.farming.ICrop;
import forestry.api.farming.IFarmHousing;
import forestry.api.farming.IFarmable;

public class FarmLogicCocoa extends FarmLogic {
	private final IFarmable cocoa = new FarmableCocoa();

	@Override
	public ItemStack getIconItemStack() {
		return new ItemStack(Items.field_151100_aR, 1, 3);
	}

	@Override
	public String getName() {
		return "Cocoa Plantation";
	}

	@Override
	public int getFertilizerConsumption() {
		return 120;
	}

	@Override
	public int getWaterConsumption(float hydrationModifier) {
		return (int) (20 * hydrationModifier);
	}

	@Override
	public boolean isAcceptedResource(ItemStack itemstack) {
		return false;
	}

	@Override
	public boolean isAcceptedGermling(ItemStack itemstack) {
		return cocoa.isGermling(itemstack);
	}

	@Override
	public boolean isAcceptedWindfall(ItemStack stack) {
		return false;
	}

	@Override
	public Collection<ItemStack> collect(World world, IFarmHousing farmHousing) {
		return null;
	}

	private final Map<BlockPos, Integer> lastExtentsCultivation = new HashMap<>();

	@Override
	public boolean cultivate(World world, IFarmHousing farmHousing, BlockPos pos, FarmDirection direction, int extent) {

		if (!lastExtentsCultivation.containsKey(pos)) {
			lastExtentsCultivation.put(pos, 0);
		}

		int lastExtent = lastExtentsCultivation.get(pos);
		if (lastExtent > extent) {
			lastExtent = 0;
		}

		BlockPos position = translateWithOffset(pos.func_177984_a(), direction, lastExtent);
		boolean result = tryPlantingCocoa(world, farmHousing, position);

		lastExtent++;
		lastExtentsCultivation.put(pos, lastExtent);

		return result;
	}

	private final Map<BlockPos, Integer> lastExtentsHarvest = new HashMap<>();

	@Override
	public Collection<ICrop> harvest(World world, BlockPos pos, FarmDirection direction, int extent) {
		if (!lastExtentsHarvest.containsKey(pos)) {
			lastExtentsHarvest.put(pos, 0);
		}

		int lastExtent = lastExtentsHarvest.get(pos);
		if (lastExtent > extent) {
			lastExtent = 0;
		}

		BlockPos position = translateWithOffset(pos.func_177984_a(), direction, lastExtent);
		Collection<ICrop> crops = getHarvestBlocks(world, position);
		lastExtent++;
		lastExtentsHarvest.put(pos, lastExtent);

		return crops;
	}

	private boolean tryPlantingCocoa(World world, IFarmHousing farmHousing, BlockPos position) {
		BlockPos.MutableBlockPos current = new BlockPos.MutableBlockPos(position);
		IBlockState blockState = world.func_180495_p(current);
		while (isJungleTreeTrunk(blockState)) {
			for (EnumFacing direction : EnumFacing.field_176754_o) {
				BlockPos candidate = new BlockPos(current.func_177958_n() + direction.func_82601_c(), current.func_177956_o(), current.func_177952_p() + direction.func_82599_e());
				if (world.func_175623_d(candidate)) {
					return farmHousing.plantGermling(cocoa, world, candidate);
				}
			}

			current.func_189536_c(EnumFacing.UP);
			if (current.func_177956_o() - position.func_177956_o() > 1) {
				break;
			}

			blockState = world.func_180495_p(current);
		}

		return false;
	}

	private static boolean isJungleTreeTrunk(IBlockState blockState) {
		Block block = blockState.func_177230_c();
		if (block == Blocks.field_150364_r && blockState.func_177229_b(BlockOldLog.field_176301_b) == BlockPlanks.EnumType.JUNGLE) {
			return true;
		}
		return false;
	}

	private Collection<ICrop> getHarvestBlocks(World world, BlockPos position) {

		Set<BlockPos> seen = new HashSet<>();
		Stack<ICrop> crops = new Stack<>();

		// Determine what type we want to harvest.
		IBlockState blockState = world.func_180495_p(position);
		Block block = blockState.func_177230_c();

		ICrop crop = null;
		if (!block.isWood(world, position)) {
			crop = cocoa.getCropAt(world, position, blockState);
			if (crop == null) {
				return crops;
			}
		}

		if (crop != null) {
			crops.add(crop);
		}

		List<BlockPos> candidates = processHarvestBlock(world, crops, seen, position, position);
		List<BlockPos> temp = new ArrayList<>();
		while (!candidates.isEmpty() && crops.size() < 20) {
			for (BlockPos candidate : candidates) {
				temp.addAll(processHarvestBlock(world, crops, seen, position, candidate));
			}
			candidates.clear();
			candidates.addAll(temp);
			temp.clear();
		}
		// Log.finest("Logic %s at %s/%s/%s has seen %s blocks.", getClass().getName(), position.x, position.y, position.z, seen.size());

		return crops;
	}

	private List<BlockPos> processHarvestBlock(World world, Stack<ICrop> crops, Set<BlockPos> seen, BlockPos start, BlockPos position) {
		List<BlockPos> candidates = new ArrayList<>();

		// Get additional candidates to return
		for (int i = -1; i < 2; i++) {
			for (int j = 0; j < 2; j++) {
				for (int k = -1; k < 2; k++) {
					BlockPos candidate = position.func_177982_a(i, j, k);
					if (candidate.equals(position)) {
						continue;
					}
					if (Math.abs(candidate.func_177958_n() - start.func_177958_n()) > 5) {
						continue;
					}
					if (Math.abs(candidate.func_177952_p() - start.func_177952_p()) > 5) {
						continue;
					}

					// See whether the given position has already been processed
					if (seen.contains(candidate)) {
						continue;
					}

					IBlockState blockState = world.func_180495_p(candidate);
					ICrop crop = cocoa.getCropAt(world, candidate, blockState);
					if (crop != null) {
						crops.push(crop);
						candidates.add(candidate);
						seen.add(candidate);
					} else if (blockState.func_177230_c().isWood(world, candidate)) {
						candidates.add(candidate);
						seen.add(candidate);
					}
				}
			}
		}

		return candidates;
	}

}
