/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.apiculture.tiles;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;

import com.google.common.base.Predicate;
import com.mojang.authlib.GameProfile;
import forestry.api.apiculture.BeeManager;
import forestry.api.apiculture.IBee;
import forestry.api.apiculture.IBeeGenome;
import forestry.api.apiculture.IBeeHousing;
import forestry.api.apiculture.IBeeHousingInventory;
import forestry.api.apiculture.IBeeListener;
import forestry.api.apiculture.IBeeModifier;
import forestry.api.apiculture.IBeekeepingLogic;
import forestry.api.apiculture.IHiveTile;
import forestry.api.apiculture.hives.IHiveRegistry;
import forestry.api.core.EnumHumidity;
import forestry.api.core.EnumTemperature;
import forestry.api.core.ForestryAPI;
import forestry.api.core.IErrorLogic;
import forestry.api.genetics.IAllele;
import forestry.apiculture.BeekeepingLogic;
import forestry.apiculture.blocks.BlockBeeHives;
import forestry.apiculture.genetics.BeeDefinition;
import forestry.apiculture.genetics.alleles.AlleleEffect;
import forestry.apiculture.network.packets.PacketActiveUpdate;
import forestry.core.config.Config;
import forestry.core.inventory.InventoryAdapter;
import forestry.core.proxy.Proxies;
import forestry.core.tiles.IActivatable;
import forestry.core.utils.DamageSourceForestry;
import forestry.core.utils.InventoryUtil;
import forestry.core.utils.ItemStackUtil;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.monster.EntityEnderman;
import net.minecraft.entity.monster.EntityPigZombie;
import net.minecraft.entity.monster.IMob;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.DamageSource;
import net.minecraft.util.EntitySelectors;
import net.minecraft.util.ITickable;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.EnumSkyBlock;
import net.minecraft.world.World;
import net.minecraft.world.biome.Biome;

public class TileHive extends TileEntity implements ITickable, IHiveTile, IActivatable, IBeeHousing {
	private static final DamageSource damageSourceBeeHive = new DamageSourceForestry("bee.hive");

	@Nonnull
	private final InventoryAdapter contained = new InventoryAdapter(2, "Contained");
	@Nonnull
	private final HiveBeeHousingInventory inventory;
	@Nonnull
	private final BeekeepingLogic beeLogic;
	@Nonnull
	private final IErrorLogic errorLogic;
	@Nonnull
	private final Predicate<EntityLivingBase> beeTargetPredicate;

	private IBee containedBee = null;
	private boolean active = false;
	private boolean angry = false;
	private int calmTime;

	/**
	 * Hack to make sure that hives glow.
	 * TODO: remove when Mojang fixes this bug: https://bugs.mojang.com/browse/MC-3329
	 */
	private boolean updatedLight;

	public TileHive() {
		inventory = new HiveBeeHousingInventory(this);
		beeLogic = new BeekeepingLogic(this);
		errorLogic = ForestryAPI.errorStateRegistry.createErrorLogic();
		beeTargetPredicate = new BeeTargetPredicate(this);
	}

	@Override
	public void func_73660_a() {
		if (Config.generateBeehivesDebug) {
			return;
		}

		if (field_145850_b.field_72995_K) {
			if (!updatedLight && field_145850_b.func_72820_D() % 100 == 0) {
				updatedLight = field_145850_b.func_180500_c(EnumSkyBlock.BLOCK, func_174877_v());
			}
			if (active && field_145850_b.field_73012_v.nextInt(4) == 0) {
				if (beeLogic.canDoBeeFX()) {
					beeLogic.doBeeFX();
				}
			}
		} else {
			boolean canWork = beeLogic.canWork(); // must be called every tick to stay updated

			if (field_145850_b.field_73012_v.nextInt(angry ? 10 : 200) == 0) {
				if (calmTime == 0) {
					if (canWork) {
						AxisAlignedBB boundingBox = AlleleEffect.getBounding(getContainedBee().getGenome(), this);
						List<EntityLivingBase> entities = field_145850_b.func_175647_a(EntityLivingBase.class, boundingBox, beeTargetPredicate);
						if (!entities.isEmpty()) {
							Collections.shuffle(entities);
							EntityLivingBase entity = entities.get(0);
							attack(entity, 2);
						}
						beeLogic.doWork();
					}
				} else {
					calmTime--;
				}
			}

			setActive(calmTime == 0);
		}
	}

	public IBee getContainedBee() {
		if (this.containedBee == null) {
			IBeeGenome beeGenome = null;
			ItemStack containedBee = contained.func_70301_a(0);
			if (containedBee != null) {
				IBee bee = BeeManager.beeRoot.getMember(containedBee);
				if (bee != null) {
					beeGenome = bee.getGenome();
				}
			}
			if (beeGenome == null) {
				beeGenome = getGenomeFromBlock();
			}
			if (beeGenome == null) {
				beeGenome = BeeDefinition.FOREST.getGenome();
			}
			this.containedBee = BeeManager.beeRoot.getBee(beeGenome);
		}
		return this.containedBee;
	}

	private IBeeGenome getGenomeFromBlock() {
		IBlockState blockState = field_145850_b.func_180495_p(field_174879_c);
		if (blockState != null) {
			Block block = blockState.func_177230_c();
			if (block instanceof BlockBeeHives) {
				IHiveRegistry.HiveType hiveType = BlockBeeHives.getHiveType(blockState);
				if (hiveType != null) {
					String speciesUid = hiveType.getSpeciesUid();
					IAllele[] template = BeeManager.beeRoot.getTemplate(speciesUid);
					if (template != null) {
						return BeeManager.beeRoot.templateAsGenome(template);
					}
				}
			}
		}
		return null;
	}

	public void setContained(@Nonnull List<ItemStack> bees) {
		for (ItemStack itemstack : bees) {
			InventoryUtil.addStack(contained, itemstack, true);
		}
	}

	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);
		contained.readFromNBT(nbttagcompound);
		beeLogic.readFromNBT(nbttagcompound);
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);
		contained.writeToNBT(nbttagcompound);
		beeLogic.writeToNBT(nbttagcompound);
		return nbttagcompound;
	}

	@Override
	public void calmBees() {
		calmTime = 5;
		angry = false;
		setActive(false);
	}

	@Override
	public boolean isAngry() {
		return angry;
	}

	@Override
	public void onAttack(World world, BlockPos pos, EntityPlayer player) {
		if (calmTime == 0) {
			angry = true;
		}
	}

	@Override
	public void onBroken(World world, BlockPos pos, EntityPlayer player, boolean canHarvest) {
		if (calmTime == 0) {
			attack(player, 10);
		}

		if (canHarvest) {
			for (ItemStack beeStack : InventoryUtil.getStacks(contained)) {
				if (beeStack != null) {
					ItemStackUtil.dropItemStackAsEntity(beeStack, field_145850_b, pos);
				}
			}
		}
	}

	private static void attack(EntityLivingBase entity, int maxDamage) {
		double attackAmount = entity.field_70170_p.field_73012_v.nextDouble() / 2.0 + 0.5;
		int damage = (int) (attackAmount * maxDamage);
		if (damage > 0) {
			// Entities are not attacked if they wear a full set of apiarist's armor.
			int count = BeeManager.armorApiaristHelper.wearsItems(entity, damageSourceBeeHive.field_76373_n, true);
			if (entity.field_70170_p.field_73012_v.nextInt(4) >= count) {
				entity.func_70097_a(damageSourceBeeHive, damage);
			}
		}
	}

	@Override
	public boolean isActive() {
		return active;
	}

	@Override
	public void setActive(boolean active) {
		if (this.active == active) {
			return;
		}
		this.active = active;

		if (!field_145850_b.field_72995_K) {
			Proxies.net.sendNetworkPacket(new PacketActiveUpdate(this), field_145850_b);
		}
	}

	@Nullable
	@Override
	public SPacketUpdateTileEntity func_189518_D_() {
		return new SPacketUpdateTileEntity(field_174879_c, 0, func_189517_E_());
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189517_E_() {
		NBTTagCompound nbt = super.func_189517_E_();
		nbt.func_74757_a("active", calmTime == 0);
		beeLogic.writeToNBT(nbt);
		return nbt;
	}

	@Override
	public void handleUpdateTag(@Nonnull NBTTagCompound tag) {
		super.handleUpdateTag(tag);
		setActive(tag.func_74767_n("active"));
		beeLogic.readFromNBT(tag);
	}

	@Override
	public void onDataPacket(NetworkManager net, SPacketUpdateTileEntity pkt) {
		super.onDataPacket(net, pkt);
		NBTTagCompound nbt = pkt.func_148857_g();
		handleUpdateTag(nbt);
	}

	@Override
	public Iterable<IBeeModifier> getBeeModifiers() {
		return Collections.emptyList();
	}

	@Override
	public Iterable<IBeeListener> getBeeListeners() {
		return Collections.emptyList();
	}

	@Nonnull
	@Override
	public IBeeHousingInventory getBeeInventory() {
		return inventory;
	}

	@Override
	public IBeekeepingLogic getBeekeepingLogic() {
		return beeLogic;
	}

	@Override
	public EnumTemperature getTemperature() {
		return EnumTemperature.getFromBiome(getBiome(), func_145831_w(), func_174877_v());
	}

	@Override
	public EnumHumidity getHumidity() {
		float humidity = ForestryAPI.climateManager.getHumidity(func_145831_w(), func_174877_v());
		return EnumHumidity.getFromValue(humidity);
	}

	@Override
	public int getBlockLightValue() {
		return func_145831_w().func_72935_r() ? 15 : 0; // hives may have the sky obstructed but should still be active
	}

	@Override
	public boolean canBlockSeeTheSky() {
		return true; // hives may have the sky obstructed but should still be active
	}

	@Override
	public World getWorldObj() {
		return field_145850_b;
	}

	@Override
	public Biome getBiome() {
		return func_145831_w().func_180494_b(func_174877_v());
	}

	@Override
	public GameProfile getOwner() {
		return null;
	}

	@Override
	public Vec3d getBeeFXCoordinates() {
		BlockPos pos = func_174877_v();
		return new Vec3d(pos.func_177958_n() + 0.5, pos.func_177956_o() + 0.25, pos.func_177952_p() + 0.5);
	}

	@Nonnull
	@Override
	public IErrorLogic getErrorLogic() {
		return errorLogic;
	}

	@Override
	public BlockPos getCoordinates() {
		return func_174877_v();
	}

	private static class BeeTargetPredicate implements Predicate<EntityLivingBase> {
		@Nonnull
		private final IHiveTile hive;

		public BeeTargetPredicate(@Nonnull IHiveTile hive) {
			this.hive = hive;
		}

		@Override
		public boolean apply(@Nullable EntityLivingBase input) {
			if (input != null && input.func_70089_S() && !input.func_82150_aj()) {
				if (input instanceof EntityPlayer) {
					return EntitySelectors.field_188444_d.apply(input);
				} else if (hive.isAngry()) {
					return true;
				} else if (input instanceof IMob) {
					// don't attack semi-passive vanilla mobs
					return !(input instanceof EntityEnderman) && !(input instanceof EntityPigZombie);
				}
			}
			return false;
		}
	}

}
