package forestry.arboriculture.blocks;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

import com.google.common.collect.Lists;
import forestry.api.arboriculture.EnumPileType;
import forestry.api.arboriculture.IAlleleTreeSpecies;
import forestry.api.arboriculture.ITree;
import forestry.api.arboriculture.IWoodProvider;
import forestry.api.arboriculture.TreeManager;
import forestry.api.core.IModelManager;
import forestry.api.core.ISpriteRegister;
import forestry.api.core.IStateMapperRegister;
import forestry.api.core.ITextureManager;
import forestry.api.core.Tabs;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IAllele;
import forestry.api.multiblock.ICharcoalPileComponent;
import forestry.apiculture.blocks.BlockCandle;
import forestry.arboriculture.PluginArboriculture;
import forestry.arboriculture.genetics.Tree;
import forestry.arboriculture.multiblock.EnumPilePosition;
import forestry.arboriculture.multiblock.ICharcoalPileControllerInternal;
import forestry.arboriculture.render.PileParticleCallback;
import forestry.arboriculture.render.PileStateMapper;
import forestry.arboriculture.tiles.TilePile;
import forestry.core.PluginCore;
import forestry.core.blocks.BlockStructure;
import forestry.core.blocks.propertys.UnlistedBlockAccess;
import forestry.core.blocks.propertys.UnlistedBlockPos;
import forestry.core.multiblock.MultiblockLogic;
import forestry.core.proxy.Proxies;
import forestry.core.render.ParticleHelper;
import forestry.core.tiles.TileUtil;
import net.minecraft.block.Block;
import net.minecraft.block.ITileEntityProvider;
import net.minecraft.block.SoundType;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.particle.ParticleManager;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Blocks;
import net.minecraft.init.Enchantments;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.util.text.TextComponentString;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.common.property.ExtendedBlockState;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.common.property.IUnlistedProperty;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public abstract class BlockPile extends BlockStructure implements ITileEntityProvider, IStateMapperRegister, ISpriteRegister {

    /**
     * B: .. T: x.
     * B: .. T: x.
     */
    protected static final AxisAlignedBB AABB_QTR_TOP_WEST = new AxisAlignedBB(0.0D, 0.5D, 0.0D, 0.5D, 1.0D, 1.0D);
    /**
     * B: .. T: .x
     * B: .. T: .x
     */
    protected static final AxisAlignedBB AABB_QTR_TOP_EAST = new AxisAlignedBB(0.5D, 0.5D, 0.0D, 1.0D, 1.0D, 1.0D);
    /**
     * B: .. T: xx
     * B: .. T: ..
     */
    protected static final AxisAlignedBB AABB_QTR_TOP_NORTH = new AxisAlignedBB(0.0D, 0.5D, 0.0D, 1.0D, 1.0D, 0.5D);
    /**
     * B: .. T: ..
     * B: .. T: xx
     */
    protected static final AxisAlignedBB AABB_QTR_TOP_SOUTH = new AxisAlignedBB(0.0D, 0.5D, 0.5D, 1.0D, 1.0D, 1.0D);
    /**
     * B: .. T: x.
     * B: .. T: ..
     */
    protected static final AxisAlignedBB AABB_OCT_TOP_NW = new AxisAlignedBB(0.0D, 0.5D, 0.0D, 0.5D, 1.0D, 0.5D);
    /**
     * B: .. T: .x
     * B: .. T: ..
     */
    protected static final AxisAlignedBB AABB_OCT_TOP_NE = new AxisAlignedBB(0.5D, 0.5D, 0.0D, 1.0D, 1.0D, 0.5D);
    /**
     * B: .. T: ..
     * B: .. T: x.
     */
    protected static final AxisAlignedBB AABB_OCT_TOP_SW = new AxisAlignedBB(0.0D, 0.5D, 0.5D, 0.5D, 1.0D, 1.0D);
    /**
     * B: .. T: ..
     * B: .. T: .x
     */
    protected static final AxisAlignedBB AABB_OCT_TOP_SE = new AxisAlignedBB(0.5D, 0.5D, 0.5D, 1.0D, 1.0D, 1.0D);
    /**
     * B: xx T: ..
     * B: xx T: ..
     */
    protected static final AxisAlignedBB AABB_SLAB_BOTTOM = new AxisAlignedBB(0.0D, 0.0D, 0.0D, 1.0D, 0.5D, 1.0D);
    
	public static final PropertyEnum<EnumPilePosition> PILE_POSITION = PropertyEnum.func_177709_a("position", EnumPilePosition.class);
	public final ParticleHelper.Callback particleCallback;
	
	public static Map<EnumPileType, BlockPile> create() {
		Map<EnumPileType, BlockPile> blockMap = new EnumMap<>(EnumPileType.class);
		for (final EnumPileType type : EnumPileType.VALUES) {
			BlockPile pile = new BlockPile() {
				
				@Override
				public String getHarvestTool(IBlockState state) {
					if(type == EnumPileType.WOOD){
						return "axe";
					}else{
						return "shovel";
					}
				}
				
				@Override
				public int getHarvestLevel(IBlockState state) {
					return 0;
				}
				
				@Override
				public EnumPileType getPileType() {
					return type;
				}
				
				@Override
				public SoundType func_185467_w() {
					if(type == EnumPileType.DIRT){
						return Blocks.field_150346_d.func_185467_w();
					}else if(type == EnumPileType.ASH){
						return Blocks.field_150354_m.func_185467_w();
					}else if(type == EnumPileType.WOOD){
						return Blocks.field_150364_r.func_185467_w();
					}
					return super.func_185467_w();
				}
			};
			blockMap.put(type, pile);
		}
		return blockMap;
	}
	
	public BlockPile() {
		super(Material.field_151578_c);
		func_149711_c(1.0F);
		func_149663_c("charcoal.pile");
		func_149647_a(Tabs.tabArboriculture);
		func_180632_j(field_176227_L.func_177621_b().func_177226_a(PILE_POSITION, EnumPilePosition.INTERIOR));
		
		particleCallback = new PileParticleCallback(this);
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		return new ExtendedBlockState(this, new IProperty[]{PILE_POSITION}, new IUnlistedProperty[]{UnlistedBlockPos.POS, UnlistedBlockAccess.BLOCKACCESS});
	}
	
	@Override
	public IBlockState getExtendedState(IBlockState state, IBlockAccess world, BlockPos pos) {
		return ((IExtendedBlockState) super.getExtendedState(state, world, pos)).withProperty(UnlistedBlockPos.POS, pos)
				.withProperty(UnlistedBlockAccess.BLOCKACCESS, world);
	}
	
    @Override
	public RayTraceResult func_180636_a(IBlockState blockState, World worldIn, BlockPos pos, Vec3d start, Vec3d end){
        List<RayTraceResult> list = Lists.newArrayList();

        for (AxisAlignedBB axisalignedbb : getCollisionBoxList(this.func_176221_a(blockState, worldIn, pos)))
        {
            list.add(this.func_185503_a(pos, start, end, axisalignedbb));
        }

        RayTraceResult raytraceresult1 = null;
        double d1 = 0.0D;

        for (RayTraceResult raytraceresult : list)
        {
            if (raytraceresult != null)
            {
                double d0 = raytraceresult.field_72307_f.func_72436_e(end);

                if (d0 > d1)
                {
                    raytraceresult1 = raytraceresult;
                    d1 = d0;
                }
            }
        }

        return raytraceresult1;
    }
	
    @Override
	public void func_185477_a(IBlockState state, World worldIn, BlockPos pos, AxisAlignedBB entityBox, List<AxisAlignedBB> collidingBoxes, Entity entityIn){
        state = this.func_176221_a(state, worldIn, pos);

        for (AxisAlignedBB axisalignedbb : getCollisionBoxList(state)){
            func_185492_a(pos, entityBox, collidingBoxes, axisalignedbb);
        }
    }

    private static List<AxisAlignedBB> getCollisionBoxList(IBlockState state){
        List<AxisAlignedBB> list = Lists.newArrayList();
        EnumPilePosition position = state.func_177229_b(PILE_POSITION);
        
        if(position != EnumPilePosition.INTERIOR){
	        list.add(AABB_SLAB_BOTTOM);
	
	        switch (position) {
				case BACK:
					list.add(AABB_QTR_TOP_SOUTH);
					break;
				case FRONT:
					 list.add(AABB_QTR_TOP_NORTH);
					break;
				case SIDE_LEFT:
					list.add(AABB_QTR_TOP_EAST);
					break;
				case SIDE_RIGHT:
					list.add(AABB_QTR_TOP_WEST);
					break;
				case CORNER_BACK_LEFT:
					list.add(AABB_OCT_TOP_NE);
					break;
				case CORNER_BACK_RIGHT:
					list.add(AABB_OCT_TOP_NW);
					break;
				case CORNER_FRONT_LEFT:
					list.add(AABB_OCT_TOP_SE);
					break;
				case CORNER_FRONT_RIGHT:
					list.add(AABB_OCT_TOP_SW);
					break;
				default:
					break;
			}
        }else{
        	list.add(Block.field_185505_j);
        }

        return list;
    }
	
	@Override
	public int func_176201_c(IBlockState state) {
		return state.func_177229_b(PILE_POSITION).ordinal();
	}
	
	@Override
	public IBlockState func_176203_a(int meta) {
		return func_176223_P().func_177226_a(PILE_POSITION, EnumPilePosition.values()[meta]);
	}
	
	@Override
	public TileEntity func_149915_a(World world, int meta) {
		return new TilePile();
	}
	
	@Override
	public int getLightValue(IBlockState state, IBlockAccess world, BlockPos pos) {
		TileEntity tile = world.func_175625_s(pos);
		if (tile instanceof ICharcoalPileComponent) {
			ICharcoalPileComponent kiln = (ICharcoalPileComponent) tile;
			if (kiln.getMultiblockLogic().isConnected() && kiln.getMultiblockLogic().getController().isAssembled() && kiln.getMultiblockLogic().getController().isActive()) {
				return 10;
			}
		}
		return super.getLightValue(state, world, pos);
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public void func_180655_c(IBlockState state, World world, BlockPos pos, Random rand) {
		TilePile pile = TileUtil.getTile(world, pos, TilePile.class);
		if(pile == null){
			return;
		}
		MultiblockLogic<ICharcoalPileControllerInternal> logic = pile.getMultiblockLogic();
		
		if (logic.isConnected() && logic.getController().isAssembled() && logic.getController().isActive() && state.func_177229_b(PILE_POSITION) != EnumPilePosition.INTERIOR) {
			float f = pos.func_177958_n() + 0.5F;
			float f1 = pos.func_177956_o() + 0.0F + rand.nextFloat() * 6.0F / 16.0F;
			float f2 = pos.func_177952_p() + 0.5F;
			float f3 = 0.52F;
			float f4 = rand.nextFloat() * 0.6F - 0.3F;
			world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, f + f3 - 0.5, f1 + 0.5, f2 + f4, 0.0D, 0.0D, 0.0D);
		}
	}

	@SideOnly(Side.CLIENT)
	@Override
	public void func_149666_a(Item item, CreativeTabs tab, List<ItemStack> subItems) {
		if (getPileType() == EnumPileType.WOOD) {
			List<ItemStack> woodPiles = new ArrayList<>();
			for (ITree tree : TreeManager.treeRoot.getIndividualTemplates()) {
				IAlleleTreeSpecies treeSpecies = tree.getGenome().getPrimary();
				ItemStack woodPile = createWoodPile(treeSpecies);
				woodPiles.add(woodPile);
			}
			subItems.addAll(woodPiles);
		} else if (getPileType() == EnumPileType.DIRT) {
			super.func_149666_a(item, tab, subItems);
		}
	}
	
	private long previousMessageTick = 0;
	
	@Override
	public boolean func_180639_a(World world, BlockPos pos, IBlockState state, EntityPlayer player, EnumHand hand, ItemStack heldItem, EnumFacing side, float hitX, float hitY, float hitZ) {
		if(world.field_72995_K){
			return false;
		}
		
		if (player.func_70093_af()) {
			return false;
		}

		TilePile pile = TileUtil.getTile(world, pos, TilePile.class);

		if(pile == null) {
			return false;
		}
		
		ICharcoalPileControllerInternal controller = pile.getMultiblockLogic().getController();

		// If the player's hands are empty and they right-click on a multiblock, they get a
		// multiblock-debugging message if the machine is not assembled.
		if (heldItem == null) {
			if (controller != null) {
				if (!controller.isAssembled()) {
					String validationError = controller.getLastValidationError();
					if (validationError != null) {
						long tick = world.func_82737_E();
						if (tick > previousMessageTick + 20) {
							player.func_145747_a(new TextComponentString(validationError));
							previousMessageTick = tick;
						}
						return true;
					}
				}
			} else {
				player.func_145747_a(new TextComponentTranslation("for.multiblock.error.notConnected"));
				return true;
			}
		}else if(BlockCandle.lightingItems.contains(heldItem.func_77973_b())) {
			if (pile.getMultiblockLogic().isConnected() && controller != null && controller.isAssembled() && !controller.isActive()) {
				controller.setActive(true);
				return true;
			}
		}
		return false;
	}
	
	/* DROP HANDLING */
	// Hack: 	When harvesting we need to get the drops in onBlockHarvested,
	// 			because Mojang destroys the block and tile before calling getDrops.
	private final ThreadLocal<List<ItemStack>> drop = new ThreadLocal<>();
	
	@Override
	public void func_176208_a(World world, BlockPos pos, IBlockState state, EntityPlayer player) {
		if (!world.field_72995_K) {
			int fortune = EnchantmentHelper.func_77506_a(Enchantments.field_185308_t, player.func_184607_cu());
			drop.set(getPileDrop(world, pos, state, fortune));
		}
	}
	
	@Nonnull
	@Override
	public List<ItemStack> getDrops(IBlockAccess world, BlockPos pos, IBlockState state, int fortune) {
		List<ItemStack> drops = drop.get();
		drop.remove();

		// not harvested, get drops normally
		if (drops == null) {
			drops = getPileDrop(world, pos, state, fortune);
		}

		return drops;
	}

	@Nonnull
	private List<ItemStack> getPileDrop(IBlockAccess world, BlockPos pos, IBlockState state, int fortune) {
		List<ItemStack> list = new ArrayList<>();
		TileEntity tile = world.func_175625_s(pos);
		if (tile instanceof ICharcoalPileComponent) {
			ICharcoalPileComponent pile = (ICharcoalPileComponent) tile;

			if (getPileType() == EnumPileType.ASH) {
				if (pile.getTreeSpecies() != null) {
					IWoodProvider woodProvider = pile.getTreeSpecies().getWoodProvider();
					ItemStack charcoal = new ItemStack(PluginArboriculture.items.charcoal, woodProvider.getCarbonization(), woodProvider.getCombustibility());
					list.add(charcoal);
					list.add(new ItemStack(PluginCore.items.ash, 3));
				} else {
					list.add(new ItemStack(Blocks.field_150346_d, 2));
					list.add(new ItemStack(PluginCore.items.ash, 2));
				}
			} else if (getPileType() == EnumPileType.DIRT) {
				list.add(new ItemStack(this));
			} else {
				list.add(createWoodPile(pile.getTreeSpecies()));
			}
		}
		return list;
	}
	
	@Override
	public void func_180663_b(World world, BlockPos pos, IBlockState state) {
		//use to override world.removeTileEntity
	}
	
	@Override
	public boolean func_149662_c(IBlockState state) {
		if(state.func_177229_b(PILE_POSITION) != EnumPilePosition.INTERIOR){
			return false;
		}
		return true;
	}
	
	@Override
	public int func_149717_k(IBlockState state) {
		if(state.func_177229_b(PILE_POSITION) != EnumPilePosition.INTERIOR){
			return 0;
		}
		return super.func_149717_k(state);
	}
	
	@Override
	public boolean func_149730_j(IBlockState state) {
		if(state.func_177229_b(PILE_POSITION) != EnumPilePosition.INTERIOR){
			return false;
		}
		return true;
	}
	
	@Override
	public boolean func_149686_d(IBlockState state) {
		if(state.func_177229_b(PILE_POSITION) != EnumPilePosition.INTERIOR){
			return false;
		}
		return true;
	}
	
	@Override
	public EnumBlockRenderType func_149645_b(IBlockState state) {
		if(state.func_177229_b(PILE_POSITION) != EnumPilePosition.INTERIOR){
			return EnumBlockRenderType.INVISIBLE;
		}
		return EnumBlockRenderType.MODEL;
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public void registerStateMapper() {
		Proxies.render.registerStateMapper(this, new PileStateMapper());
	}

	@Override
	public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player) {
		if(getPileType() == EnumPileType.WOOD){
			TileEntity tile = world.func_175625_s(pos);
			if (tile instanceof ICharcoalPileComponent) {
				return createWoodPile(((ICharcoalPileComponent) tile).getTreeSpecies());
			}
		}else if(getPileType() == EnumPileType.DIRT){
			return super.getPickBlock(state, target, world, pos, player);
		}
		return null;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public void registerModel(Item item, IModelManager manager) {
		if(getPileType() == EnumPileType.WOOD){
			manager.registerItemModel(item, 0, "woodPile");
		}else if(getPileType() == EnumPileType.DIRT){
			manager.registerItemModel(item, 0, "dirtPile");
		}else if(getPileType() == EnumPileType.ASH){
			manager.registerItemModel(item, 0, "ashPile");
		}
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public void registerSprites(ITextureManager manager) {
		for(IAllele allele : AlleleManager.alleleRegistry.getRegisteredAlleles().values()){
			if(allele instanceof IAlleleTreeSpecies){
				IAlleleTreeSpecies treeSpecies = (IAlleleTreeSpecies) allele;
				treeSpecies.getWoodProvider().registerSprites(Item.func_150898_a(this), manager);
			}
		}
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public boolean addHitEffects(IBlockState state, World worldObj, RayTraceResult target, ParticleManager effectRenderer) {
		return ParticleHelper.addBlockHitEffects(worldObj, target.func_178782_a(), target.field_178784_b, effectRenderer, particleCallback);
	}

	@SideOnly(Side.CLIENT)
	@Override
	public boolean addDestroyEffects(World world, BlockPos pos, ParticleManager effectRenderer) {
		IBlockState blockState = world.func_180495_p(pos);
		return ParticleHelper.addDestroyEffects(world, this, blockState, pos, effectRenderer, particleCallback);
	}

	public static IAlleleTreeSpecies getTreeSpecies(ItemStack stack) {
		if (stack != null && stack.func_77942_o()) {
			NBTTagCompound tagCompound = stack.func_77978_p();

			// legacy
			if (tagCompound.func_74764_b("ContainedTree")) {
				ITree tree = new Tree(tagCompound.func_74775_l("ContainedTree"));
				return tree.getGenome().getPrimary();
			}

			String treeSpeciesUid = tagCompound.func_74779_i("TreeSpecies");
			return (IAlleleTreeSpecies) AlleleManager.alleleRegistry.getAllele(treeSpeciesUid);
		}
		return null;
	}
	
	@Override
	public int getFlammability(IBlockAccess world, BlockPos pos, EnumFacing face) {
		if(getPileType() == EnumPileType.WOOD){
			TileEntity tile = world.func_175625_s(pos);
			if (tile instanceof ICharcoalPileComponent) {
				IAlleleTreeSpecies tree = ((ICharcoalPileComponent) tile).getTreeSpecies();
				if (tree != null) {
					ItemStack wood = tree.getWoodProvider().getWoodStack();
					if(wood != null){
						Block block = Block.func_149634_a(wood.func_77973_b());
						return block.getFlammability(world, pos, face);
					}
				}
			}
		}
		return super.getFlammability(world, pos, face);
	}

	@Override
	public int getFireSpreadSpeed(IBlockAccess world, BlockPos pos, EnumFacing face) {
		if(getPileType() == EnumPileType.WOOD){
			TileEntity tile = world.func_175625_s(pos);
			if (tile instanceof ICharcoalPileComponent) {
				IAlleleTreeSpecies tree = ((ICharcoalPileComponent) tile).getTreeSpecies();
				if(tree != null){
					ItemStack wood = tree.getWoodProvider().getWoodStack();
					if(wood != null){
						Block block = Block.func_149634_a(wood.func_77973_b());
						return block.getFireSpreadSpeed(world, pos, face);
					}
				}
			}
		}
		return super.getFireSpreadSpeed(world, pos, face);
		
	}
	
	@Override
	public float func_176195_g(IBlockState blockState, World world, BlockPos pos) {
		if(getPileType() == EnumPileType.WOOD){
			TileEntity tile = world.func_175625_s(pos);
			if (tile instanceof ICharcoalPileComponent) {

				IAlleleTreeSpecies tree = ((ICharcoalPileComponent) tile).getTreeSpecies();
				if(tree != null){
					ItemStack wood = tree.getWoodProvider().getWoodStack();
					if (wood != null) {
						Block block = Block.func_149634_a(wood.func_77973_b());
						return block.func_176203_a(wood.func_77960_j()).func_185887_b(world, pos);
					}
				}
			}
		}
		return super.func_176195_g(blockState, world, pos);
	}

	public static ItemStack createWoodPile(IAlleleTreeSpecies treeSpecies) {
		if (treeSpecies != null) {
			ItemStack stack = new ItemStack(PluginArboriculture.blocks.piles.get(EnumPileType.WOOD));
			NBTTagCompound nbtItem = new NBTTagCompound();
			nbtItem.func_74778_a("TreeSpecies", treeSpecies.getUID());
			stack.func_77982_d(nbtItem);
			return stack;
		}
		return null;
	}
	
	public abstract EnumPileType getPileType();
}
