/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.item.ItemStack;

import forestry.api.core.EnumCamouflageType;
import forestry.api.greenhouse.IGreenhouseAccess;
import forestry.core.utils.Log;

public class GreenhouseAccess implements IGreenhouseAccess {

	private final Map<ItemStack, Float> greenhouseGlasses = new HashMap<>();
	private final Map<EnumCamouflageType, List<ItemStack>> camouflageBlockBlacklist = new EnumMap<>(EnumCamouflageType.class);
	
	public GreenhouseAccess() {
		for (EnumCamouflageType type : EnumCamouflageType.VALUES) {
			camouflageBlockBlacklist.put(type, new ArrayList<>());
		}
	}
	
	@Override
	public void registerGreenhouseGlass(@Nonnull ItemStack glass, float lightTransmittance) {
		if (glass == null || glass.func_77973_b() == null) {
			Log.error("Fail to register greenhouse glass, because it is null");
			return;
		}
		Block block = Block.func_149634_a(glass.func_77973_b());
		if (block == null) {
			Log.error("Fail to register greenhouse glass, it has no matching block: " + glass + ".");
			return;
		}
		IBlockState defaultBlockState = block.func_176223_P();
		if (block.func_149662_c(defaultBlockState)) {
			Log.error("Fail to register greenhouse glass, it is opaque: " + block + ".");
			return;
		}
		for (ItemStack greenhouseGlass : greenhouseGlasses.keySet()) {
			if (greenhouseGlass.func_77973_b() == glass.func_77973_b() && greenhouseGlass.func_77952_i() == glass.func_77952_i() && ItemStack.func_77970_a(glass, greenhouseGlass)) {
				Log.error("Fail to register greenhouse glass, because it is already registered: " + glass + ".");
				return;
			}
		}
		greenhouseGlasses.put(glass, lightTransmittance);
	}
	
	@Override
	public float getGreenhouseGlassLightTransmittance(@Nonnull ItemStack glass) {
		if (glass == null || glass.func_77973_b() == null || Block.func_149634_a(glass.func_77973_b()) == null) {
			return 0.5F;
		}
		for (Map.Entry<ItemStack, Float> greenhouseGlassEntry : greenhouseGlasses.entrySet()) {
			ItemStack greenhouseGlass = greenhouseGlassEntry.getKey();
			if (greenhouseGlass.func_77973_b() == glass.func_77973_b() && greenhouseGlass.func_77952_i() == glass.func_77952_i() && ItemStack.func_77970_a(glass, greenhouseGlass)) {
				return greenhouseGlassEntry.getValue();
			}
		}
		return 0.5F;
	}

	@Override
	public boolean isGreenhouseGlass(@Nonnull ItemStack glass) {
		if (glass == null || glass.func_77973_b() == null || Block.func_149634_a(glass.func_77973_b()) == null) {
			return false;
		}
		for (ItemStack greenhouseGlass : greenhouseGlasses.keySet()) {
			if (greenhouseGlass.func_77973_b() == glass.func_77973_b() && greenhouseGlass.func_77952_i() == glass.func_77952_i() && ItemStack.func_77970_a(glass, greenhouseGlass)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public void addToCamouflageBlockBlackList(@Nonnull EnumCamouflageType type, @Nonnull ItemStack camouflageBlock) {
		if (camouflageBlock == null || camouflageBlock.func_77973_b() == null) {
			Log.error("Fail to add camouflage block item to the black list, because it is null");
			return;
		}
		Block block = Block.func_149634_a(camouflageBlock.func_77973_b());
		if (block == null) {
			Log.error("Fail to add camouflage block item to the black list: because it has no block.");
			return;
		}
		for (ItemStack camouflageBlacklisted : camouflageBlockBlacklist.get(type)) {
			if (camouflageBlacklisted.func_77973_b() == camouflageBlock.func_77973_b() && camouflageBlacklisted.func_77952_i() == camouflageBlock.func_77952_i() && ItemStack.func_77970_a(camouflageBlock, camouflageBlacklisted)) {
				Log.error("Fail to add camouflage block item to the black list, because it is already registered: " + camouflageBlock + ".");
				return;
			}
		}
		camouflageBlockBlacklist.get(type).add(camouflageBlock);
	}

	@Override
	public boolean isOnCamouflageBlockBlackList(@Nonnull EnumCamouflageType type, @Nonnull ItemStack camouflageBlock) {
		if (camouflageBlock == null || camouflageBlock.func_77973_b() == null || Block.func_149634_a(camouflageBlock.func_77973_b()) == null) {
			return false;
		}
		for (ItemStack camouflageBlacklisted : camouflageBlockBlacklist.get(type)) {
			if (camouflageBlacklisted.func_77973_b() == camouflageBlock.func_77973_b() && camouflageBlacklisted.func_77952_i() == camouflageBlock.func_77952_i() && ItemStack.func_77970_a(camouflageBlock, camouflageBlacklisted)) {
				return true;
			}
		}
		return false;
	}

}
