package binnie.core.machines;

import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.block.ITileEntityProvider;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyInteger;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.gui.GuiScreen;
import net.minecraft.client.util.ITooltipFlag;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.world.World;

import net.minecraftforge.fluids.FluidActionResult;
import net.minecraftforge.fluids.FluidUtil;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;
import net.minecraftforge.fluids.capability.IFluidHandler;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.core.utils.Translator;

import binnie.core.BinnieCore;
import binnie.core.machines.component.IRender;
import binnie.core.util.TileUtil;

class BlockMachine extends Block implements IBlockMachine, ITileEntityProvider {
	public static final PropertyInteger MACHINE_TYPE = PropertyInteger.func_177719_a("machine_type", 0, 15);
	private MachineGroup group;

	public BlockMachine(MachineGroup group, String blockName) {
		super(Material.field_151573_f);
		this.group = group;
		this.func_149711_c(1.5f);
		this.setRegistryName(blockName);
	}

	@Override
	public void func_149666_a(CreativeTabs tab, NonNullList<ItemStack> itemList) {
		for (final MachinePackage pack : this.group.getPackages()) {
			if (pack.isActive()) {
				itemList.add(new ItemStack(this, 1, pack.getMetadata()));
			}
		}
	}

	@Override
	public boolean func_149662_c(IBlockState state) {
		return false;
	}

	@Override
	public boolean func_149686_d(IBlockState state) {
		return false;
	}

	@Override
	public EnumBlockRenderType func_149645_b(IBlockState state) {
		return EnumBlockRenderType.MODEL;
	}

	@Override
	public boolean canRenderInLayer(IBlockState state, BlockRenderLayer layer) {
		return layer == BlockRenderLayer.CUTOUT;
	}

	@Override
	public int func_176201_c(IBlockState state) {
		return state.func_177229_b(MACHINE_TYPE);
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		return func_176223_P().func_177226_a(MACHINE_TYPE, meta);
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		return new BlockStateContainer(this, MACHINE_TYPE);
	}

	@Override
	@Nullable
	public MachinePackage getPackage(final int meta) {
		return this.group.getPackage(meta);
	}

	@Override
	public String getMachineName(final int meta) {
		MachinePackage machinePackage = this.getPackage(meta);
		return (machinePackage == null) ? "Unnamed Machine" : machinePackage.getDisplayName();
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_190948_a(ItemStack stack, @Nullable World player, List<String> tooltip, ITooltipFlag advanced) {
		super.func_190948_a(stack, player, tooltip, advanced);
		if (GuiScreen.func_146272_n()) {
			MachinePackage machinePackage = this.getPackage(stack.func_77960_j());
			if(machinePackage != null){
				tooltip.add(machinePackage.getInformation());
			}
		} else {
			tooltip.add(TextFormatting.ITALIC + "<" + Translator.translateToLocal("for.gui.tooltip.tmi") + ">");
		}
	}

	@Override
	public int func_180651_a(IBlockState state) {
		return state.func_177229_b(MACHINE_TYPE);
	}

	@Override
	public TileEntity func_149915_a(final World world, final int meta) {
		MachinePackage machinePackage = this.group.getPackage(meta);
		if (machinePackage == null) {
			machinePackage = this.group.getPackage(0);
			Preconditions.checkNotNull(machinePackage, "Machine has no packages %s", this);
		}
		return machinePackage.createTileEntity();
	}

	@Override
	public boolean func_180639_a(World worldIn, BlockPos pos, IBlockState state, EntityPlayer playerIn, EnumHand hand, EnumFacing facing, float hitX, float hitY, float hitZ) {
		if (!BinnieCore.getBinnieProxy().isSimulating(worldIn)) {
			return true;
		}
		if (playerIn.func_70093_af()) {
			return true;
		}
		TileEntity tileEntity = worldIn.func_175625_s(pos);
		if (tileEntity != null) {
			IFluidHandler tileFluidHandler = tileEntity.getCapability(CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY, facing);
			if (tileFluidHandler != null && FluidUtil.interactWithFluidHandler(playerIn, hand, tileFluidHandler)) {
				return true;
			}
			if (tileEntity instanceof TileEntityMachine) {
				((TileEntityMachine) tileEntity).getMachine().onRightClick(worldIn, playerIn, pos);
			}
		}
		return true;
	}

	@Override
	public void func_180633_a(World world, BlockPos pos, IBlockState state, EntityLivingBase entityliving, ItemStack stack) {
		super.func_180633_a(world, pos, state, entityliving, stack);
		if (!BinnieCore.getBinnieProxy().isSimulating(world)) {
			return;
		}
		final IMachine machine = Machine.getMachine(world.func_175625_s(pos));
		if (machine == null) {
			return;
		}
		if (entityliving instanceof EntityPlayer) {
			machine.setOwner(((EntityPlayer) entityliving).func_146103_bH());
		}
	}

	@Override
	public void func_180663_b(World world, BlockPos pos, IBlockState state) {
		final TileEntityMachine entity = TileUtil.getTile(world, pos, TileEntityMachine.class);
		if (entity != null) {
			entity.onBlockDestroy();
		}
		super.func_180663_b(world, pos, state);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_180655_c(IBlockState state, World world, BlockPos pos, Random rand) {
		if (world.func_175667_e(pos)) {
			IMachine machine = Machine.getMachine(TileUtil.getTile(world, pos, TileEntity.class));
			if (machine != null) {
				for (IRender.RandomDisplayTick renders : machine.getInterfaces(IRender.RandomDisplayTick.class)) {
					renders.onRandomDisplayTick(world, pos, rand);
				}
			}
		}
	}

	@Override
	public boolean removedByPlayer(IBlockState state, World world, BlockPos pos, EntityPlayer player, boolean willHarvest) {
		if (!player.field_71075_bZ.field_75098_d) {
			return super.removedByPlayer(state, world, pos, player, willHarvest);
		}
		return world.func_175698_g(pos);
	}
}
