package binnie.botany.gardening;

import javax.annotation.Nullable;
import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.block.SoundType;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.init.Blocks;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

import net.minecraftforge.common.IPlantable;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.core.IItemModelRegister;
import forestry.api.core.IModelManager;

import binnie.botany.Botany;
import binnie.botany.CreativeTabBotany;
import binnie.botany.api.EnumAcidity;
import binnie.botany.api.EnumMoisture;
import binnie.botany.api.EnumSoilType;
import binnie.botany.api.IBlockSoil;
import binnie.botany.core.BotanyCore;
import binnie.core.util.I18N;

public class BlockSoil extends Block implements IBlockSoil, IItemModelRegister {
	public static final PropertyEnum<EnumMoisture> MOISTURE = PropertyEnum.func_177709_a("moisture", EnumMoisture.class);
	public static final PropertyEnum<EnumAcidity> ACIDITY = PropertyEnum.func_177709_a("acidity", EnumAcidity.class);
	public static final AxisAlignedBB SOIL_BLOCK_AABB = new AxisAlignedBB(0.0D, 0.0D, 0.0D, 1.0D, 0.9375D, 1.0D);
	private final boolean weedKilled;
	private final EnumSoilType type;

	public BlockSoil(EnumSoilType type, String blockName, boolean weedKilled) {
		super(Material.field_151578_c);
		this.type = type;
		this.weedKilled = weedKilled;
		func_149663_c("botany.soil." + type.func_176610_l());
		func_149647_a(CreativeTabBotany.instance);
		setRegistryName(blockName);
		func_149675_a(true);
		func_149713_g(255);
		func_149711_c(0.5f);
		func_149672_a(SoundType.field_185849_b);
	}

	public static int getMeta(EnumAcidity acid, EnumMoisture moisture) {
		return acid.ordinal() * 3 + moisture.ordinal();
	}

	public static String getPH(ItemStack stack, boolean withColor, boolean byNeutralNone) {
		EnumAcidity ph = EnumAcidity.values()[stack.func_77952_i() / 3];
		if (byNeutralNone) {
			if (ph == EnumAcidity.NEUTRAL) {
				return "";
			}
		}
		return TextFormatting.GRAY + I18N.localise("botany.ph") + ": " + ph.getLocalisedName(withColor);
	}

	public static String getMoisture(ItemStack stack, boolean withColor, boolean byNormalNone) {
		EnumMoisture moisure = EnumMoisture.values()[stack.func_77952_i() % 3];
		if (byNormalNone) {
			if (moisure == EnumMoisture.NORMAL) {
				return "";
			}
		}
		return TextFormatting.GRAY + I18N.localise("botany.moisture") + ": " + moisure.getLocalisedName(withColor);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public boolean func_176225_a(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		switch (side) {
			case UP:
				return true;

			case NORTH:
			case SOUTH:
			case WEST:
			case EAST:
				IBlockState iblockstate = blockAccess.func_180495_p(pos.func_177972_a(side));
				Block block = iblockstate.func_177230_c();
				return !iblockstate.func_185914_p() && block != Blocks.field_150458_ak && block != Blocks.field_185774_da && block != this;
		}
		return super.func_176225_a(blockState, blockAccess, pos, side);
	}

	/**
	 * Used to determine ambient occlusion and culling when rebuilding chunks for render
	 */
	@Override
	public boolean func_149662_c(IBlockState state) {
		return false;
	}

	@Override
	public boolean func_149686_d(IBlockState state) {
		return false;
	}

	@Override
	public String func_149739_a() {
		return super.func_149739_a().replaceFirst("tile.", "");
	}

	@Override
	public AxisAlignedBB func_185496_a(IBlockState state, IBlockAccess source, BlockPos pos) {
		return SOIL_BLOCK_AABB;
	}

	@Nullable
	@Override
	public AxisAlignedBB func_180646_a(IBlockState blockState, IBlockAccess worldIn, BlockPos pos) {
		return SOIL_BLOCK_AABB;
	}

	@Override
	public AxisAlignedBB func_180640_a(IBlockState state, World worldIn, BlockPos pos) {
		return SOIL_BLOCK_AABB;
	}

	@Override
	public void func_149666_a(CreativeTabs tab, NonNullList<ItemStack> list) {
		for (int i = 0; i < 9; ++i) {
			list.add(new ItemStack(this, 1, i));
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void registerModel(Item item, IModelManager manager) {
		for (EnumAcidity acidity : EnumAcidity.values()) {
			for (EnumMoisture moisture : EnumMoisture.values()) {
				String modelName = "";
				if (acidity != EnumAcidity.NEUTRAL) {
					modelName += acidity.func_176610_l();
				}

				if (moisture != EnumMoisture.NORMAL) {
					if (!modelName.isEmpty()) {
						modelName += "_";
					}
					modelName += moisture.func_176610_l();
				}

				if (modelName.isEmpty()) {
					modelName = "normal";
				}

				String identifier;
				if (weedKilled) {
					identifier = type.func_176610_l() + "_no_weed/" + modelName;
				} else {
					identifier = type.func_176610_l() + "/" + modelName;
				}
				manager.registerItemModel(item, moisture.ordinal() + acidity.ordinal() * 3, identifier);
			}
		}
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		return new BlockStateContainer(this, MOISTURE, ACIDITY);
	}

	@Override
	public int func_176201_c(IBlockState state) {
		return getMeta(state.func_177229_b(ACIDITY), state.func_177229_b(MOISTURE));
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		EnumMoisture moisture = EnumMoisture.values()[meta % 3];
		EnumAcidity acidity = EnumAcidity.values()[meta / 3];
		return func_176223_P().func_177226_a(MOISTURE, moisture).func_177226_a(ACIDITY, acidity);
	}

	public EnumSoilType getType() {
		return type;
	}

	@Override
	public int func_180651_a(IBlockState state) {
		return func_176201_c(state);
	}

	@Override
	public void func_180650_b(World world, BlockPos pos, IBlockState state, Random rand) {
		EnumMoisture moisture = state.func_177229_b(MOISTURE);
		EnumMoisture desiredMoisture = BotanyCore.getGardening().getNaturalMoisture(world, pos);
		if (desiredMoisture.ordinal() > moisture.ordinal()) {
			moisture = ((moisture == EnumMoisture.DRY) ? EnumMoisture.NORMAL : EnumMoisture.DAMP);
		} else if (desiredMoisture.ordinal() < moisture.ordinal()) {
			moisture = ((moisture == EnumMoisture.DAMP) ? EnumMoisture.NORMAL : EnumMoisture.DRY);
		}

		IBlockState blockState = state.func_177226_a(MOISTURE, moisture);
		if (state != blockState) {
			world.func_180501_a(pos, blockState, 2);
		}

		if (!weedKilled) {
			if (rand.nextInt(5 - getType(world, pos).ordinal()) != 0) {
				return;
			}
			pos = pos.func_177984_a();
			if (!world.func_175623_d(pos)) {
				return;
			}
			world.func_180501_a(pos, Botany.gardening().plant.func_176203_a(PlantType.WEEDS.ordinal()), 2);
		}
	}

	@Override
	public void func_189540_a(IBlockState state, World worldIn, BlockPos pos, Block blockIn, BlockPos fromPos) {
		super.func_189540_a(state, worldIn, pos, blockIn, fromPos);

		if (worldIn.func_180495_p(pos.func_177984_a()).func_185904_a().func_76220_a()) {
			worldIn.func_175656_a(pos, Blocks.field_150346_d.func_176223_P());
		}
	}

	@Override
	public EnumAcidity getPH(World world, BlockPos pos) {
		return world.func_180495_p(pos).func_177229_b(ACIDITY);
	}

	@Override
	public EnumMoisture getMoisture(World world, BlockPos pos) {
		return world.func_180495_p(pos).func_177229_b(MOISTURE);
	}

	@Override
	public EnumSoilType getType(World world, BlockPos pos) {
		return type;
	}

	@Override
	public boolean fertilise(World world, BlockPos pos, EnumSoilType maxLevel) {
		EnumSoilType type = getType(world, pos);
		if (type.ordinal() >= maxLevel.ordinal()) {
			return false;
		}
		IBlockState old = world.func_180495_p(pos);
		return world.func_180501_a(pos, BotanyCore.getGardening().getSoilBlock(maxLevel, weedKilled).func_176223_P().func_177226_a(ACIDITY, old.func_177229_b(ACIDITY)).func_177226_a(MOISTURE, old.func_177229_b(MOISTURE)), 2);
	}

	@Override
	public boolean degrade(World world, BlockPos pos, EnumSoilType minLevel) {
		EnumSoilType type = getType(world, pos);
		if (type.ordinal() <= minLevel.ordinal()) {
			return false;
		}
		IBlockState old = world.func_180495_p(pos);
		return world.func_180501_a(pos, BotanyCore.getGardening().getSoilBlock(minLevel, weedKilled).func_176223_P().func_177226_a(ACIDITY, old.func_177229_b(ACIDITY)).func_177226_a(MOISTURE, old.func_177229_b(MOISTURE)), 2);
	}

	@Override
	public boolean setPH(World world, BlockPos pos, EnumAcidity pH) {
		IBlockState blockState = world.func_180495_p(pos);
		return world.func_175656_a(pos, blockState.func_177226_a(ACIDITY, pH));
	}

	@Override
	public boolean setMoisture(World world, BlockPos pos, EnumMoisture moisture) {
		IBlockState blockState = world.func_180495_p(pos);
		return world.func_175656_a(pos, blockState.func_177226_a(MOISTURE, moisture));
	}

	@Override
	public void onNeighborChange(IBlockAccess world, BlockPos pos, BlockPos neighbor) {
		super.onNeighborChange(world, pos, neighbor);
		if (world.isSideSolid(pos.func_177984_a(), EnumFacing.DOWN, false)) {
			((World) world).func_180501_a(pos, Blocks.field_150346_d.func_176223_P(), 2);
		}
	}

	@Override
	public boolean canSustainPlant(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing direction, IPlantable plantable) {
		IBlockState plant = plantable.getPlant(world, pos.func_177984_a());
		if (plant.func_177230_c() == Botany.flower) {
			return true;
		}

		if (plant.func_177230_c() == Botany.gardening().plant) {
			return !weedKilled || !BlockPlant.isWeed(world, pos);
		}
		return world instanceof World && Blocks.field_150346_d.canSustainPlant(state, world, pos, direction, plantable);
	}

	@Override
	public boolean resistsWeeds(World world, BlockPos pos) {
		return weedKilled;
	}

	public boolean isWeedKilled() {
		return weedKilled;
	}
}
