package binnie.botany.farm;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import net.minecraft.block.Block;
import net.minecraft.client.renderer.texture.TextureMap;
import net.minecraft.init.Blocks;
import net.minecraft.item.Item;
import net.minecraft.item.ItemBlock;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.NonNullList;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import net.minecraftforge.fluids.FluidStack;

import forestry.api.farming.FarmDirection;
import forestry.api.farming.ICrop;
import forestry.api.farming.IFarmHousing;
import forestry.api.farming.IFarmable;
import forestry.core.owner.IOwnedTile;
import forestry.core.utils.BlockUtil;

import binnie.Binnie;
import binnie.botany.Botany;
import binnie.botany.api.EnumAcidity;
import binnie.botany.api.EnumMoisture;
import binnie.botany.api.EnumSoilType;
import binnie.botany.api.IBlockSoil;
import binnie.botany.api.IGardeningManager;
import binnie.botany.core.BotanyCore;
import binnie.botany.flower.TileEntityFlower;
import binnie.core.liquid.ManagerLiquid;

public class GardenLogic extends FarmLogic {
	List<IFarmable> farmables;
	private EnumMoisture moisture;
	@Nullable
	private EnumAcidity acidity;
	private boolean fertilised;
	private String name;
	private NonNullList<ItemStack> produce;
	private ItemStack icon;

	public GardenLogic(EnumMoisture moisture2, @Nullable EnumAcidity acidity2, boolean isManual, boolean isFertilised, ItemStack icon2, String name2) {
		this.isManual = isManual;
		produce = NonNullList.func_191196_a();
		farmables = new ArrayList<>();
		farmables.add(new FarmableFlower());
		farmables.add(new FarmableVanillaFlower());

		moisture = moisture2;
		acidity = acidity2;
		fertilised = isFertilised;
		icon = icon2;
		name = name2;
	}

	@Override
	public int getFertilizerConsumption() {
		return fertilised ? 8 : 2;
	}

	@Override
	public int getWaterConsumption(float hydrationModifier) {
		return (int) (moisture.ordinal() * 40 * hydrationModifier);
	}

	@Override
	public boolean isAcceptedResource(ItemStack itemstack) {
		IGardeningManager gardening = BotanyCore.getGardening();
		return gardening.isSoil(itemstack.func_77973_b()) ||
				itemstack.func_77973_b() == Item.func_150898_a(Blocks.field_150354_m) ||
				itemstack.func_77973_b() == Item.func_150898_a(Blocks.field_150346_d) ||
				gardening.isAcidFertiliser(itemstack) ||
				gardening.isAlkalineFertiliser(itemstack);
	}

	@Override
	public NonNullList<ItemStack> collect(World world, IFarmHousing farmHousing) {
		NonNullList<ItemStack> products = produce;
		produce = NonNullList.func_191196_a();
		return products;
	}

	@Override
	public boolean cultivate(World world, IFarmHousing farmHousing, BlockPos pos, FarmDirection direction, int extent) {
		return maintainSoil(world, pos, direction, extent, farmHousing) ||
				(!isManual && maintainWater(world, pos, direction, extent, farmHousing)) ||
				maintainCrops(world, pos.func_177984_a(), direction, extent, farmHousing);
	}

	private boolean isWaste(ItemStack stack) {
		return stack.func_77973_b() == Item.func_150898_a(Blocks.field_150346_d);
	}

	private boolean maintainSoil(World world, BlockPos pos, FarmDirection direction, int extent, IFarmHousing housing) {
		IGardeningManager gardening = BotanyCore.getGardening();
		for (int i = 0; i < extent; ++i) {
			BlockPos position = pos.func_177967_a(direction.getFacing(), i);
			if (fertilised && gardening.isSoil(getBlock(world, position))) {
				IBlockSoil soil = (IBlockSoil) getBlock(world, position);
				if (soil.fertilise(world, position, EnumSoilType.FLOWERBED)) {
					continue;
				}
			}

			if (getBlock(world, position.func_177984_a()) == Botany.plant) {
				world.func_175698_g(position.func_177984_a());
			} else {
				if (acidity != null && gardening.isSoil(getBlock(world, position))) {
					IBlockSoil soil = (IBlockSoil) getBlock(world, position);
					EnumAcidity pH = soil.getPH(world, position);
					if (pH.ordinal() < acidity.ordinal()) {
						ItemStack stack = getAvailableAlkaline(housing);
						if (!stack.func_190926_b() && soil.setPH(world, position, EnumAcidity.values()[pH.ordinal() + 1])) {
							NonNullList<ItemStack> resources = NonNullList.func_191196_a();
							resources.add(stack);
							housing.getFarmInventory().removeResources(resources);
							continue;
						}
					}
					if (pH.ordinal() > acidity.ordinal()) {
						ItemStack stack = getAvailableAcid(housing);
						if (!stack.func_190926_b() && soil.setPH(world, position, EnumAcidity.values()[pH.ordinal() - 1])) {
							NonNullList<ItemStack> resources = NonNullList.func_191196_a();
							resources.add(stack);
							housing.getFarmInventory().removeResources(resources);
							continue;
						}
					}
				}

				if (!isAirBlock(world, position) && !BlockUtil.isReplaceableBlock(getBlockState(world, position), world, position)) {
					ItemStack block = getAsItemStack(world, position);
					ItemStack loam = getAvailableLoam(housing);
					if (isWaste(block) && !loam.func_190926_b()) {
						produce.addAll(Blocks.field_150346_d.getDrops(world, position, Block.func_149634_a(block.func_77973_b()).func_176203_a(block.func_77952_i()), 0));
						setBlock(world, position, Blocks.field_150350_a, 0);
						return trySetSoil(world, position, loam, housing);
					}
				} else if (!isManual) {
					if (!isWaterBlock(world, position)) {
						if (i % 2 == 0) {
							return trySetSoil(world, position, housing);
						}
						FarmDirection cclock = FarmDirection.EAST;
						if (direction == FarmDirection.EAST) {
							cclock = FarmDirection.SOUTH;
						} else if (direction == FarmDirection.SOUTH) {
							cclock = FarmDirection.EAST;
						} else if (direction == FarmDirection.WEST) {
							cclock = FarmDirection.SOUTH;
						}
						BlockPos previous = position.func_177972_a(cclock.getFacing());
						ItemStack soil2 = getAsItemStack(world, previous);
						if (!gardening.isSoil(soil2.func_77973_b())) {
							trySetSoil(world, position, housing);
						}
					}
				}
			}
		}
		return false;
	}

	private boolean maintainWater(World world, BlockPos pos, FarmDirection direction, int extent, IFarmHousing housing) {
		for (int i = 0; i < extent; ++i) {
			BlockPos position = pos.func_177967_a(direction.getFacing(), i);
			if (!isAirBlock(world, position) && !BlockUtil.isReplaceableBlock(getBlockState(world, position), world, position)) {
				continue;
			}
			if (isWaterBlock(world, position)) {
				continue;
			}

			boolean isEnclosed = true;
			if (world.func_175623_d(position.func_177974_f())) {
				isEnclosed = false;
			} else if (world.func_175623_d(position.func_177976_e())) {
				isEnclosed = false;
			} else if (world.func_175623_d(position.func_177968_d())) {
				isEnclosed = false;
			} else if (world.func_175623_d(position.func_177978_c())) {
				isEnclosed = false;
			}

			isEnclosed = (isEnclosed || moisture != EnumMoisture.DAMP);
			if (isEnclosed) {
				return trySetWater(world, position, housing);
			}
		}
		return false;
	}

	private ItemStack getAvailableLoam(IFarmHousing housing) {
		EnumMoisture[] moistures;
		if (moisture == EnumMoisture.DAMP) {
			moistures = new EnumMoisture[]{
					EnumMoisture.DAMP,
					EnumMoisture.NORMAL,
					EnumMoisture.DRY
			};
		} else if (moisture == EnumMoisture.DRY) {
			moistures = new EnumMoisture[]{
					EnumMoisture.DRY,
					EnumMoisture.DAMP,
					EnumMoisture.DRY
			};
		} else {
			moistures = new EnumMoisture[]{
					EnumMoisture.DRY,
					EnumMoisture.NORMAL,
					EnumMoisture.DAMP
			};
		}

		EnumAcidity[] acidities = {
				EnumAcidity.NEUTRAL,
				EnumAcidity.ACID,
				EnumAcidity.ALKALINE
		};

		for (EnumMoisture moist : moistures) {
			for (EnumAcidity acid : acidities) {
				for (Block type : new Block[]{Botany.flowerbed, Botany.loam, Botany.soil}) {
					int meta = acid.ordinal() * 3 + moist.ordinal();
					NonNullList<ItemStack> resources = NonNullList.func_191196_a();
					ItemStack resourceStack = new ItemStack(type, 1, meta);
					resources.add(resourceStack);
					if (housing.getFarmInventory().hasResources(resources)) {
						return resourceStack;
					}
				}
			}
		}

		NonNullList<ItemStack> resources = NonNullList.func_191196_a();
		ItemStack resourceStack = new ItemStack(Blocks.field_150346_d);
		resources.add(resourceStack);
		if (housing.getFarmInventory().hasResources(resources)) {
			return new ItemStack(Blocks.field_150346_d);
		}
		return ItemStack.field_190927_a;
	}

	private boolean trySetSoil(World world, BlockPos position, IFarmHousing housing) {
		return trySetSoil(world, position, getAvailableLoam(housing), housing);
	}

	private boolean trySetSoil(World world, BlockPos position, ItemStack loam, IFarmHousing housing) {
		ItemStack copy = loam;
		if (loam.func_190926_b()) {
			return false;
		}

		if (loam.func_77973_b() == Item.func_150898_a(Blocks.field_150346_d)) {
			loam = new ItemStack(Botany.soil, 1, 4);
		}

		setBlock(world, position, ((ItemBlock) loam.func_77973_b()).func_179223_d(), loam.func_77952_i());
		NonNullList<ItemStack> resources = NonNullList.func_191196_a();
		resources.add(copy);
		housing.getFarmInventory().removeResources(resources);
		return true;
	}

	private boolean trySetWater(World world, BlockPos position, IFarmHousing housing) {
		FluidStack water = Binnie.LIQUID.getFluidStack(ManagerLiquid.WATER, 1000);
		if (moisture == EnumMoisture.DAMP) {
			if (water == null || !housing.hasLiquid(water)) {
				return false;
			}
			setBlock(world, position, Blocks.field_150355_j, 0);
			housing.removeLiquid(water);
			return true;
		}

		if (moisture != EnumMoisture.DRY) {
			return trySetSoil(world, position, housing);
		}
		ItemStack sand = new ItemStack(Blocks.field_150354_m, 1);
		NonNullList<ItemStack> resources = NonNullList.func_191196_a();
		resources.add(sand);
		if (!housing.getFarmInventory().hasResources(resources)) {
			return false;
		}
		setBlock(world, position, Blocks.field_150354_m, 0);
		housing.getFarmInventory().removeResources(resources);
		return true;
	}

	public void setIcon(ItemStack icon) {
		this.icon = icon;
	}

	@Override
	public boolean isAcceptedGermling(ItemStack itemstack) {
		for (IFarmable farmable : farmables) {
			if (farmable.isGermling(itemstack)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public Collection<ICrop> harvest(World world, BlockPos pos, FarmDirection direction, int extent) {
		if (isManual) {
			return Collections.emptyList();
		}
		return farmables.stream().map(farmable -> farmable.getCropAt(world, pos.func_177984_a(), world.func_180495_p(pos.func_177984_a())))
				.filter(Objects::nonNull).collect(Collectors.toList());
	}

	@Override
	public ResourceLocation getTextureMap() {
		return TextureMap.field_110575_b;
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public ItemStack getIconItemStack() {
		return icon;
	}

	protected boolean maintainCrops(World world, BlockPos pos, FarmDirection direction, int extent, IFarmHousing housing) {
		IGardeningManager gardening = BotanyCore.getGardening();
		for (int i = 0; i < extent; ++i) {
			BlockPos position = pos.func_177967_a(direction.getFacing(), i);
			if (isAirBlock(world, position) || BlockUtil.isReplaceableBlock(getBlockState(world, position), world, position)) {
				ItemStack below = getAsItemStack(world, position.func_177977_b());
				if (gardening.isSoil(below.func_77973_b())) {
					return trySetCrop(world, position, housing);
				}
			}
		}
		return false;
	}

	private boolean trySetCrop(World world, BlockPos position, IFarmHousing housing) {
		for (IFarmable farmable : farmables) {
			if (!housing.plantGermling(farmable, world, position)) {
				continue;
			}

			if (housing instanceof IOwnedTile) {
				TileEntity tile = world.func_175625_s(position);
				if (tile instanceof TileEntityFlower) {
					TileEntityFlower flower = (TileEntityFlower) tile;
					IOwnedTile owned = (IOwnedTile) housing;

					flower.setOwner(owned.getOwnerHandler().getOwner());
				}
			}
			return true;
		}
		return false;
	}

	public ItemStack getAvailableAcid(IFarmHousing housing) {
		for (ItemStack stack : BotanyCore.getGardening().getAcidFertilisers()) {
			if (!stack.func_190926_b()) {
				NonNullList<ItemStack> resources = NonNullList.func_191196_a();
				resources.add(stack);
				if (housing.getFarmInventory().hasResources(resources)) {
					return stack;
				}
			}
		}
		return ItemStack.field_190927_a;
	}

	public ItemStack getAvailableAlkaline(IFarmHousing housing) {
		for (ItemStack stack : BotanyCore.getGardening().getAlkalineFertilisers()) {
			if (!stack.func_190926_b()) {
				NonNullList<ItemStack> resources = NonNullList.func_191196_a();
				resources.add(stack);
				if (housing.getFarmInventory().hasResources(resources)) {
					return stack;
				}
			}
		}
		return ItemStack.field_190927_a;
	}
}
