package binnie.core.machines;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import net.minecraft.block.state.IBlockState;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.IFluidTank;

import binnie.core.BinnieCore;
import binnie.core.machines.inventory.IChargedSlots;
import binnie.core.machines.power.IPoweredMachine;
import binnie.core.machines.power.IProcess;
import binnie.core.machines.power.ITankMachine;
import binnie.core.machines.power.PowerSystem;
import binnie.core.util.ItemStackSet;

public class MachineUtil {
	private IMachine machine;

	public MachineUtil(final IMachine machine) {
		this.machine = machine;
	}

	public IInventory getInventory() {
		return this.machine.getInterface(IInventory.class);
	}

	public ITankMachine getTankContainer() {
		return this.machine.getInterface(ITankMachine.class);
	}

	public IPoweredMachine getPoweredMachine() {
		return this.machine.getInterface(IPoweredMachine.class);
	}

	public boolean isSlotEmpty(final int slot) {
		return this.getInventory().func_70301_a(slot).func_190926_b();
	}

	public IFluidTank getTank(final int id) {
		return this.getTankContainer().getTanks()[id];
	}

	public boolean spaceInTank(final int id, final int amount) {
		final IFluidTank tank = this.getTank(id);
		final int space = tank.getCapacity() - tank.getFluidAmount();
		return amount <= space;
	}

	public ItemStack getStack(final int slot) {
		return this.getInventory().func_70301_a(slot);
	}

	public void deleteStack(final int slot) {
		this.setStack(slot, ItemStack.field_190927_a);
	}

	public ItemStack decreaseStack(final int slot, final int amount) {
		return this.getInventory().func_70298_a(slot, amount);
	}

	public void setStack(final int slot, final ItemStack stack) {
		this.getInventory().func_70299_a(slot, stack);
	}

	public void fillTank(final int id, final FluidStack liquidStack) {
		final IFluidTank tank = this.getTank(id);
		tank.fill(liquidStack, true);
	}

	public void addStack(final int slot, final ItemStack addition) {
		final ItemStack merge = this.getStack(slot);
		if (merge.func_190926_b()) {
			this.setStack(slot, addition);
		} else {
			if (merge.func_77969_a(addition) && merge.func_190916_E() + addition.func_190916_E() <= merge.func_77976_d()) {
				merge.func_190917_f(addition.func_190916_E());
				this.setStack(slot, merge);
			}
		}
	}

	@Nullable
	public FluidStack drainTank(final int tank, final int amount) {
		return this.getTank(tank).drain(amount, true);
	}

	public boolean liquidInTank(final int tankIndex, final int amount) {
		IFluidTank tank = this.getTank(tankIndex);
		FluidStack drain = tank.drain(amount, false);
		return drain != null && drain.amount == amount;
	}

	public void damageItem(final ItemStack item, final int slot, final int damage) {
		if (damage < 0) {
			item.func_77964_b(Math.max(0, item.func_77952_i() + damage));
		} else if (item.func_96631_a(damage, new Random())) {
			this.setStack(slot, ItemStack.field_190927_a);
		}
		this.setStack(slot, item);
	}

	public boolean isTankEmpty(final int tankInput) {
		return this.getTank(tankInput).getFluidAmount() == 0;
	}

	@Nullable
	public FluidStack getFluid(final int tankInput) {
		return this.getTank(tankInput).getFluid();
	}

	public ItemStack[] getStacks(final int[] slots) {
		return getStacks(slots, false);
	}

	public ItemStack[] getStacks(final int[] slots, boolean copy) {
		final ItemStack[] stacks = new ItemStack[slots.length];
		for (int i = 0; i < slots.length; ++i) {
			ItemStack stack = this.getStack(slots[i]);
			if (copy) {
				stack = stack.func_77946_l();
			}
			stacks[i] = stack;
		}
		return stacks;
	}

	public boolean hasIngredients(final int[] recipe, final int[] inventory) {
		final ItemStackSet requiredStacks = new ItemStackSet();
		Collections.addAll(requiredStacks, this.getStacks(recipe));
		final ItemStackSet inventoryStacks = new ItemStackSet();
		Collections.addAll(inventoryStacks, this.getStacks(inventory));
		requiredStacks.removeAll(inventoryStacks);
		return requiredStacks.isEmpty();
	}

	public boolean removeIngredients(final int[] recipe, final int[] inventorySlots) {
		if (!hasIngredients(recipe, inventorySlots)) {
			return false;
		}

		List<ItemStack> requiredStacks = this.getNonEmptyStacks(recipe, true);
		for (ItemStack requiredStack : requiredStacks) {
			IInventory inventory = this.getInventory();
			for (int slot : inventorySlots) {
				ItemStack stackInSlot = this.getStack(slot);
				if (!stackInSlot.func_190926_b() && ItemStack.func_179545_c(requiredStack, stackInSlot) && ItemStack.func_77970_a(requiredStack, stackInSlot)) {
					if (requiredStack.func_190916_E() >= stackInSlot.func_190916_E()) {
						requiredStack.func_190918_g(stackInSlot.func_190916_E());
						inventory.func_70304_b(slot);
					} else {
						stackInSlot.func_190918_g(requiredStack.func_190916_E());
						requiredStack.func_190920_e(0);
						break;
					}
				}
			}
		}

		return true;
	}

	public void useEnergyMJ(final float powerUsage) {
		this.getPoweredMachine().getInterface().useEnergy(PowerSystem.MJ, powerUsage, true);
	}

	public boolean hasEnergyMJ(final float powerUsage) {
		return this.getPoweredMachine().getInterface().useEnergy(PowerSystem.MJ, powerUsage, false) >= powerUsage;
	}

	public float getSlotCharge(final int slot) {
		return this.machine.getInterface(IChargedSlots.class).getCharge(slot);
	}

	public void useCharge(final int slot, final float loss) {
		this.machine.getInterface(IChargedSlots.class).alterCharge(slot, -loss);
	}

	public Random getRandom() {
		return new Random();
	}

	public void refreshBlock() {
		BlockPos pos = machine.getTileEntity().func_174877_v();
		World world = machine.getWorld();
		IBlockState blockState = world.func_180495_p(pos);
		machine.getWorld().func_184138_a(pos, blockState, blockState, 0);
	}

	public IProcess getProcess() {
		return this.machine.getInterface(IProcess.class);
	}

	public NonNullList<ItemStack> getNonEmptyStacks(final int[] slots) {
		return getNonEmptyStacks(slots, false);
	}

	public NonNullList<ItemStack> getNonEmptyStacks(final int[] slots, boolean copy) {
		final NonNullList<ItemStack> stacks = NonNullList.func_191196_a();
		for (final ItemStack stack : this.getStacks(slots, copy)) {
			if (!stack.func_190926_b()) {
				stacks.add(stack);
			}
		}
		return stacks;
	}

	public boolean isServer() {
		return BinnieCore.getBinnieProxy().isSimulating(this.machine.getWorld());
	}
}
