package binnie.botany.flower;

import javax.annotation.Nullable;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTUtil;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraft.world.biome.Biome;

import com.mojang.authlib.GameProfile;

import net.minecraftforge.common.EnumPlantType;

import forestry.api.core.EnumHumidity;
import forestry.api.core.EnumTemperature;
import forestry.api.core.IErrorState;
import forestry.api.genetics.IIndividual;
import forestry.api.genetics.IPollinatable;
import forestry.api.lepidopterology.IButterfly;
import forestry.api.lepidopterology.IButterflyNursery;

import binnie.Binnie;
import binnie.botany.Botany;
import binnie.botany.api.EnumFlowerStage;
import binnie.botany.api.EnumSoilType;
import binnie.botany.api.IAlleleFlowerSpecies;
import binnie.botany.api.IFlower;
import binnie.botany.api.IFlowerColor;
import binnie.botany.api.IFlowerGenome;
import binnie.botany.api.IFlowerRoot;
import binnie.botany.api.IFlowerType;
import binnie.botany.api.IGardeningManager;
import binnie.botany.core.BotanyCore;
import binnie.botany.gardening.PlantType;
import binnie.botany.genetics.EnumFlowerColor;
import binnie.botany.genetics.EnumFlowerType;
import binnie.botany.genetics.Flower;
import binnie.core.BinnieCore;

public class TileEntityFlower extends TileEntity implements IPollinatable, IButterflyNursery {
	@Nullable
	IFlower flower;
	@Nullable
	GameProfile owner;
	int section;
	@Nullable
	RenderInfo renderInfo;
	@Nullable
	IButterfly caterpillar;
	int matureTime;

	public TileEntityFlower() {
		flower = null;
		section = 0;
		renderInfo = null;
		matureTime = 0;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbtCompound) {
		if (nbtCompound.func_74764_b("Flower")) {
			flower = new Flower(nbtCompound.func_74775_l("Flower"));
			if (flower.getAge() == 0) {
				flower.setFlowered(false);
			}
		}

		if (nbtCompound.func_74764_b("section")) {
			section = nbtCompound.func_74771_c("section");
		}

		if (nbtCompound.func_74764_b("owner")) {
			owner = NBTUtil.func_152459_a(nbtCompound.func_74775_l("owner"));
		}

		if (nbtCompound.func_74764_b("CATER") && BinnieCore.isLepidopteryActive()) {
			matureTime = nbtCompound.func_74762_e("caterTime");
			caterpillar = Binnie.GENETICS.getButterflyRoot().getMember(nbtCompound.func_74775_l("cater"));
		}

		readRenderInfo(nbtCompound);
		super.func_145839_a(nbtCompound);
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbtCompound) {
		if (flower != null) {
			nbtCompound.func_74782_a("Flower", flower.writeToNBT(new NBTTagCompound()));
		}

		if (owner != null) {
			NBTTagCompound nbt = new NBTTagCompound();
			NBTUtil.func_180708_a(nbt, owner);
			nbtCompound.func_74782_a("owner", nbt);
		}

		if (caterpillar != null) {
			nbtCompound.func_74768_a("caterTime", matureTime);
			NBTTagCompound subcompound = new NBTTagCompound();
			caterpillar.writeToNBT(subcompound);
			nbtCompound.func_74782_a("cater", subcompound);
		}

		nbtCompound.func_74774_a("section", (byte) getSection());
		return super.func_189515_b(nbtCompound);
	}

	public void create(ItemStack stack, @Nullable GameProfile owner) {
		IFlower flower = BotanyCore.getFlowerRoot().getMember(stack);
		create(flower, owner);
	}

	public void create(IFlower flower, @Nullable GameProfile owner) {
		this.flower = flower;
		if (flower.getAge() == 0) {
			flower.setFlowered(false);
		}

		updateRender(true);
		this.owner = owner;
	}

	@Override
	public EnumPlantType getPlantType() {
		return EnumPlantType.Plains;
	}

	@Override
	@Nullable
	public IIndividual getPollen() {
		return getFlower();
	}

	@Override
	public boolean canMateWith(IIndividual individual) {
		return isBreeding()
				&& individual instanceof IFlower
				&& getFlower() != null
				&& getFlower().getMate() == null
				&& getFlower().hasFlowered()
				&& !getFlower().isGeneticEqual(individual);
	}

	@Override
	public void mateWith(IIndividual individual) {
		if (getFlower() == null || !(individual instanceof IFlower)) {
			return;
		}

		IAlleleFlowerSpecies primary = (IAlleleFlowerSpecies) individual.getGenome().getPrimary();
		IAlleleFlowerSpecies primary2 = getFlower().getGenome().getPrimary();
		if (primary == primary2 || field_145850_b.field_73012_v.nextInt(4) == 0) {
			getFlower().mate((IFlower) individual);
			field_145850_b.func_175704_b(field_174879_c, field_174879_c);
		}
	}

	@Nullable
	public IFlower getFlower() {
		if (getSection() <= 0) {
			return flower;
		}

		TileEntity tile = field_145850_b.func_175625_s(field_174879_c.func_177979_c(getSection()));
		if (tile instanceof TileEntityFlower) {
			return ((TileEntityFlower) tile).getFlower();
		}
		return null;
	}

	public boolean isBreeding() {
		Block roots = func_145831_w().func_180495_p(func_174877_v().func_177977_b()).func_177230_c();
		return BotanyCore.getGardening().isSoil(roots);
	}

	public void randomUpdate(Random rand) {
		if (field_145850_b.func_180495_p(field_174879_c).func_177230_c() != Botany.flower) {
			func_145843_s();
			return;
		}
		if (getSection() > 0) {
			return;
		}
		if (flower == null) {
			return;
		}

		// TODO always false?
		if (flower.getGenome() == null) {
			func_145843_s();
			return;
		}
		if (!isBreeding()) {
			return;
		}

		float light = field_145850_b.func_175699_k(field_174879_c);
		if (light < 6.0f) {
			for (int dx = -2; dx <= 2; ++dx) {
				for (int dy = -2; dy <= 2; ++dy) {
					light -= (field_145850_b.func_175710_j(field_174879_c) ? 0.0f : 0.5f);
				}
			}
		}

		IGardeningManager gardening = BotanyCore.getGardening();
		boolean canTolerate = gardening.canTolerate(getFlower(), field_145850_b, field_174879_c);
		EnumSoilType soil = gardening.getSoilType(field_145850_b, field_174879_c.func_177977_b());
		if (rand.nextFloat() < getFlower().getGenome().getAgeChance()) {
			if (flower.getAge() < 1) {
				if (canTolerate && light > 6.0f) {
					doFlowerAge();
				}
			} else {
				doFlowerAge();
			}
		}

		if (canTolerate && flower.getAge() > 1 && !flower.isWilted() && light > 6.0f) {
			flower.setFlowered(true);
		}

		if (!canTolerate && flower.isWilted() && rand.nextInt(2 + Math.max(flower.getAge(), 2)) == 0) {
			kill();
			return;
		}

		if (light < 2.0f && flower.isWilted()) {
			kill();
			return;
		}

		if (!canTolerate || light < 1.0f) {
			flower.setWilted(true);
		} else {
			flower.setWilted(false);
		}

		float CHANCE_DISPERSAL = 0.8f;
		CHANCE_DISPERSAL += 0.2f * flower.getGenome().getFertility();
		CHANCE_DISPERSAL *= 1.0f + soil.ordinal() * 0.5f;
		float CHANCE_POLLINATE = 1.0f;
		CHANCE_POLLINATE += 0.25f * flower.getGenome().getFertility();
		CHANCE_POLLINATE *= 1.0f + soil.ordinal() * 0.5f;
		float CHANCE_SELFPOLLINATE = 0.2f * CHANCE_POLLINATE;
		if (field_145850_b.field_73012_v.nextFloat() < CHANCE_DISPERSAL && flower.hasFlowered() && !flower.isWilted()) {
			IFlowerGenome mate = flower.getMate();
			if (mate != null) {
				boolean dispersed = false;
				for (int a = 0; a < 5 && !dispersed; ++a) {
					int dx2;
					int dz;
					for (dx2 = 0, dz = 0; dx2 == 0 && dz == 0; dx2 = rand.nextInt(3) - 1, dz = rand.nextInt(3) - 1) {
					}

					Block b2 = field_145850_b.func_180495_p(field_174879_c.func_177982_a(dx2, -1, dz)).func_177230_c();
					if (field_145850_b.func_175623_d(field_174879_c.func_177982_a(dx2, 0, dz)) && BotanyCore.getGardening().isSoil(b2)) {
						IFlower offspring = flower.getOffspring(field_145850_b, field_174879_c);
						if (offspring != null) {
							BotanyCore.getFlowerRoot().plant(field_145850_b, field_174879_c.func_177982_a(dx2, 0, dz), offspring, getOwner());
							flower.removeMate();
							dispersed = true;
						}
					}
				}
			}
		}

		if (field_145850_b.field_73012_v.nextFloat() < CHANCE_POLLINATE && flower.hasFlowered() && !flower.isWilted()) {
			for (int a2 = 0; a2 < 4; ++a2) {
				int dx3;
				int dz2;
				for (dx3 = 0, dz2 = 0; dx3 == 0 && dz2 == 0; dx3 = rand.nextInt(5) - 2, dz2 = rand.nextInt(5) - 2) {
				}
				TileEntity tile = field_145850_b.func_175625_s(field_174879_c.func_177982_a(dx3, 0, dz2));
				if (tile instanceof IPollinatable && ((IPollinatable) tile).canMateWith(getFlower())) {
					((IPollinatable) tile).mateWith(getFlower());
				}
			}
		}

		if (field_145850_b.field_73012_v.nextFloat() < CHANCE_SELFPOLLINATE && flower.hasFlowered() && flower.getMate() == null) {
			mateWith(getFlower());
		}
		spawnButterflies();
		matureCaterpillar();
		checkIfDead(false);
		updateRender(true);
	}

	private void doFlowerAge() {
		getFlower().age();
		if (getFlower().getAge() == 1) {
			IFlowerRoot flowerRoot = BotanyCore.getFlowerRoot();
			flowerRoot.onGrowFromSeed(field_145850_b, field_174879_c);
			if (getOwner() != null && getFlower() != null) {
				flowerRoot.getBreedingTracker(field_145850_b, getOwner()).registerBirth(getFlower());
			}
		}
	}

	private NBTTagCompound writeRenderInfo(NBTTagCompound tag) {
		if (renderInfo != null) {
			NBTTagCompound nbtRenderInfo = new NBTTagCompound();
			nbtRenderInfo.func_74774_a("primary", (byte) renderInfo.primary.getID());
			nbtRenderInfo.func_74774_a("secondary", (byte) renderInfo.secondary.getID());
			nbtRenderInfo.func_74774_a("stem", (byte) renderInfo.stem.getID());
			nbtRenderInfo.func_74774_a("type", (byte) renderInfo.type.ordinal());
			nbtRenderInfo.func_74774_a("age", renderInfo.age);
			nbtRenderInfo.func_74774_a("section", renderInfo.section);
			nbtRenderInfo.func_74757_a("wilted", renderInfo.wilted);
			nbtRenderInfo.func_74757_a("flowered", renderInfo.flowered);
			tag.func_74782_a("renderinfo", nbtRenderInfo);
		}
		return tag;
	}

	private void readRenderInfo(NBTTagCompound tag) {
		if (tag.func_74764_b("renderinfo")) {
			NBTTagCompound infotag = tag.func_74775_l("renderinfo");
			RenderInfo info = new RenderInfo();
			info.primary = EnumFlowerColor.values()[infotag.func_74771_c("primary")].getFlowerColorAllele();
			info.secondary = EnumFlowerColor.values()[infotag.func_74771_c("secondary")].getFlowerColorAllele();
			info.stem = EnumFlowerColor.values()[infotag.func_74771_c("stem")].getFlowerColorAllele();
			info.type = EnumFlowerType.values()[infotag.func_74771_c("type")];
			info.age = infotag.func_74771_c("age");
			info.section = infotag.func_74771_c("section");
			info.wilted = infotag.func_74767_n("wilted");
			info.flowered = infotag.func_74767_n("flowered");
			setRender(info);
		}
	}

	@Override
	public void onDataPacket(NetworkManager net, SPacketUpdateTileEntity pkt) {
		super.onDataPacket(net, pkt);
		// TODO client only call?
		readRenderInfo(pkt.func_148857_g());
	}

	@Override
	public NBTTagCompound func_189517_E_() {
		updateRender(true);
		return writeRenderInfo(super.func_189517_E_());
	}

	@Override
	public SPacketUpdateTileEntity func_189518_D_() {
		// TODO getFlower().getGenome() != null always true?
		if (renderInfo == null && getFlower() != null && getFlower().getGenome() != null) {
			renderInfo = new RenderInfo(getFlower(), this);
		}
		return (renderInfo != null) ? new SPacketUpdateTileEntity(field_174879_c, 0, func_189517_E_()) : null;
	}

	public void updateRender(boolean update) {
		if (update && getFlower() != null && getFlower().getGenome() != null) {
			RenderInfo newInfo = new RenderInfo(getFlower(), this);
			if (renderInfo == null || !newInfo.equals(renderInfo)) {
				setRender(newInfo);
			}
		}

		TileEntity above = field_145850_b.func_175625_s(field_174879_c.func_177984_a());
		if (above instanceof TileEntityFlower) {
			((TileEntityFlower) above).updateRender(true);
		}
	}

	public int getSection() {
		return section;
	}

	public void setSection(int i) {
		section = i;
		if (BinnieCore.getBinnieProxy().isSimulating(field_145850_b)) {
			updateRender(true);
		}
	}

	public ItemStack getItemStack() {
		IFlower flower = getFlower();
		if (flower == null) {
			return ItemStack.field_190927_a;
		}
		return Binnie.GENETICS.getFlowerRoot().getMemberStack(flower, EnumFlowerStage.getStage(flower));
	}

	@Nullable
	private TileEntityFlower getRoot() {
		if (getSection() == 0) {
			return null;
		}
		TileEntity tile = field_145850_b.func_175625_s(field_174879_c.func_177979_c(getSection()));
		return (tile instanceof TileEntityFlower) ? ((TileEntityFlower) tile) : null;
	}

	public void onShear() {
		if (getRoot() != null) {
			getRoot().onShear();
		}

		if (getFlower() == null || getFlower().getAge() <= 1) {
			return;
		}

		Random rand = new Random();
		IFlower cutting = (IFlower) getFlower().copy();
		cutting.setAge(0);
		ItemStack cuttingStack = BotanyCore.getFlowerRoot().getMemberStack(cutting, EnumFlowerStage.SEED);
		float f = 0.7f;
		double d = rand.nextFloat() * f + (1.0f - f) * 0.5;
		double d2 = rand.nextFloat() * f + (1.0f - f) * 0.5;
		double d3 = rand.nextFloat() * f + (1.0f - f) * 0.5;
		EntityItem entityitem = new EntityItem(field_145850_b, field_174879_c.func_177958_n() + d, field_174879_c.func_177956_o() + d2, field_174879_c.func_177952_p() + d3, cuttingStack);
		entityitem.func_174867_a(10);
		field_145850_b.func_72838_d(entityitem);
		for (int maxAge = getFlower().getMaxAge(), i = 0; i < maxAge; ++i) {
			if (rand.nextBoolean()) {
				getFlower().age();
				if (checkIfDead(true)) {
					return;
				}
			}
		}
	}

	public boolean checkIfDead(boolean wasCut) {
		if (getSection() != 0) {
			return getRoot().checkIfDead(wasCut);
		}

		EnumSoilType soil = BotanyCore.getGardening().getSoilType(field_145850_b, field_174879_c);
		int maxAge = (int) (flower.getMaxAge() * (1.0f + soil.ordinal() * 0.25f));
		if (flower.getAge() > maxAge) {
			if (!wasCut && flower.getMate() != null) {
				field_145850_b.func_175698_g(field_174879_c);
				IFlower offspring = flower.getOffspring(field_145850_b, field_174879_c.func_177977_b());
				TileEntity above = field_145850_b.func_175625_s(field_174879_c.func_177984_a());
				if (above instanceof TileEntityFlower) {
					field_145850_b.func_175698_g(field_174879_c.func_177984_a());
				}
				BotanyCore.getFlowerRoot().plant(field_145850_b, field_174879_c, offspring, getOwner());
			} else {
				kill();
			}
			return true;
		}
		return false;
	}

	public void kill() {
		if (flower.getAge() > 0) {
			field_145850_b.func_180501_a(field_174879_c, Botany.plant.func_176203_a(PlantType.DEAD_FLOWER.ordinal()), 2);
		} else {
			field_145850_b.func_175698_g(field_174879_c);
		}

		for (int i = 1; field_145850_b.func_175625_s(field_174879_c.func_177981_b(i)) instanceof TileEntityFlower; ++i) {
			field_145850_b.func_175698_g(field_174879_c.func_177981_b(i));
		}
	}

	public boolean onBonemeal() {
		if (getFlower() == null) {
			return false;
		}
		if (!isBreeding()) {
			return false;
		}
		if (getFlower().isWilted()) {
			return false;
		}

		//this.doFlowerAge();
		if (getFlower().getAge() > 1 && !getFlower().hasFlowered()) {
			getFlower().setFlowered(true);
		}
		checkIfDead(false);
		updateRender(true);
		return true;
	}

	@Nullable
	public GameProfile getOwner() {
		return owner;
	}

	public void setOwner(GameProfile ownerName) {
		owner = ownerName;
	}

	public void spawnButterflies() {
		/*if (!BinnieCore.isLepidopteryActive()) {
			return;
		}
		final int x = this.xCoord;
		final int y = this.yCoord;
		final int z = this.zCoord;
		final World world = this.worldObj;
		if (this.getCaterpillar() != null) {
			return;
		}
		if (world.rand.nextFloat() >= this.getFlower().getGenome().getSappiness()) {
			return;
		}
		if (this.worldObj.rand.nextFloat() >= 0.2f) {
			return;
		}
		final IButterfly spawn = Binnie.Genetics.getButterflyRoot().getIndividualTemplates().get(this.worldObj.rand.nextInt(Binnie.Genetics.getButterflyRoot().getIndividualTemplates().size()));
		if (this.worldObj.rand.nextFloat() >= spawn.getGenome().getPrimary().getRarity() * 0.5f) {
			return;
		}
		if (this.worldObj.countEntities(EntityButterfly.class) > 100) {
			return;
		}
		if (!spawn.canSpawn(this.worldObj, x, y, z)) {
			return;
		}
		if (world.isAirBlock(x, y + 1, z)) {
			this.attemptButterflySpawn(world, spawn, x, y + 1, z);
		}
		else if (world.isAirBlock(x - 1, y, z)) {
			this.attemptButterflySpawn(world, spawn, x - 1, y, z);
		}
		else if (world.isAirBlock(x + 1, y, z)) {
			this.attemptButterflySpawn(world, spawn, x + 1, y, z);
		}
		else if (world.isAirBlock(x, y, z - 1)) {
			this.attemptButterflySpawn(world, spawn, x, y, z - 1);
		}
		else if (world.isAirBlock(x, y, z + 1)) {
			this.attemptButterflySpawn(world, spawn, x, y, z + 1);
		}*/
	}

	private void attemptButterflySpawn(World world, IButterfly butterfly, double x, double y, double z) {
		if (BinnieCore.isLepidopteryActive()) {
			Binnie.GENETICS.getButterflyRoot().spawnButterflyInWorld(world, butterfly.copy(), x, y + 0.10000000149011612, z);
		}
	}

	public GameProfile getOwnerName() {
		return getOwner();
	}

	@Override
	public EnumTemperature getTemperature() {
		return EnumTemperature.getFromValue(field_145850_b.func_180494_b(func_174877_v()).func_185353_n());
	}

	@Override
	public EnumHumidity getHumidity() {
		return EnumHumidity.getFromValue(field_145850_b.func_180494_b(func_174877_v()).func_76727_i());
	}

	public void setErrorState(int state) {
	}

	public int getErrorOrdinal() {
		return 0;
	}

	public boolean addProduct(ItemStack product, boolean all) {
		return false;
	}

	@Override
	@Nullable
	public IButterfly getCaterpillar() {
		return caterpillar;
	}

	@Override
	public void setCaterpillar(@Nullable IButterfly butterfly) {
		caterpillar = butterfly;
		matureTime = 0;
	}

	@Override
	public IIndividual getNanny() {
		return getFlower();
	}

	@Override
	public boolean canNurse(IButterfly butterfly) {
		return getFlower() != null && !getFlower().isWilted() && getFlower().getAge() > 1;
	}

	private void matureCaterpillar() {
		//TODO Spawn the caterpillar in a cocon on the next Tree or other thing
		/*if (this.getCaterpillar() == null) {
			return;
		}
        ++this.matureTime;
		if (this.matureTime >= this.caterpillar.getGenome().getLifespan() / (this.caterpillar.getGenome().getFertility() * 2) && this.caterpillar.canTakeFlight(this.worldObj, this.xCoord, this.yCoord, this.zCoord)) {
			if (this.worldObj.isAirBlock(this.xCoord, this.yCoord + 1, this.zCoord)) {
				this.attemptButterflySpawn(this.worldObj, this.caterpillar, this.xCoord, this.yCoord + 1, this.zCoord);
			}
			else if (this.worldObj.isAirBlock(this.xCoord - 1, this.yCoord, this.zCoord)) {
				this.attemptButterflySpawn(this.worldObj, this.caterpillar, this.xCoord - 1, this.yCoord, this.zCoord);
			}
			else if (this.worldObj.isAirBlock(this.xCoord + 1, this.yCoord, this.zCoord)) {
				this.attemptButterflySpawn(this.worldObj, this.caterpillar, this.xCoord + 1, this.yCoord, this.zCoord);
			}
			else if (this.worldObj.isAirBlock(this.xCoord, this.yCoord, this.zCoord - 1)) {
				this.attemptButterflySpawn(this.worldObj, this.caterpillar, this.xCoord, this.yCoord, this.zCoord - 1);
			}
			else if (this.worldObj.isAirBlock(this.xCoord, this.yCoord, this.zCoord + 1)) {
				this.attemptButterflySpawn(this.worldObj, this.caterpillar, this.xCoord, this.yCoord, this.zCoord + 1);
			}
			this.setCaterpillar(null);
		}*/
	}

	public void setRender(RenderInfo render) {
		renderInfo = render;
		section = renderInfo.section;
		if (!field_145850_b.field_72995_K) {
			IBlockState blockState = field_145850_b.func_180495_p(field_174879_c);
			field_145850_b.func_184138_a(field_174879_c, blockState, blockState, 0);
		} else {
			field_145850_b.func_175704_b(field_174879_c, field_174879_c);
		}
	}

	public int getAge() {
		return (renderInfo == null) ? 1 : renderInfo.age;
	}

	public int getRenderSection() {
		return (renderInfo == null) ? 1 : renderInfo.section;
	}

	public boolean isWilted() {
		return renderInfo != null && renderInfo.wilted;
	}

	public boolean isFlowered() {
		return renderInfo == null || renderInfo.flowered;
	}

	public int getPrimaryColour() {
		return (renderInfo == null)
				? EnumFlowerColor.Red.getFlowerColorAllele().getColor(false)
				: renderInfo.primary.getColor(isWilted());
	}

	public int getSecondaryColour() {
		return (renderInfo == null)
				? EnumFlowerColor.Red.getFlowerColorAllele().getColor(false)
				: renderInfo.secondary.getColor(isWilted());
	}

	public int getStemColour() {
		return (renderInfo == null)
				? EnumFlowerColor.Green.getFlowerColorAllele().getColor(false)
				: renderInfo.stem.getColor(isWilted());
	}

	public IFlowerType getType() {
		return (renderInfo == null)
				? EnumFlowerType.Poppy
				: renderInfo.type;
	}

	@Override
	public Biome getBiome() {
		return func_145831_w().func_180494_b(func_174877_v());
	}

	@Nullable
	public IErrorState getErrorState() {
		return null;
	}

	public void setErrorState(IErrorState state) {
	}

	public boolean setErrorCondition(boolean condition, IErrorState errorState) {
		return false;
	}

	public Set<IErrorState> getErrorStates() {
		return new HashSet<>();
	}

	@Override
	public BlockPos getCoordinates() {
		return func_174877_v();
	}

	@Override
	public boolean isPollinated() {
		return isBreeding();
	}

	@Override
	public World getWorldObj() {
		return field_145850_b;
	}

	public static class RenderInfo {
		public IFlowerColor primary;
		public IFlowerColor secondary;
		public IFlowerColor stem;
		public IFlowerType type;
		public byte age;
		public boolean wilted;
		public boolean flowered;
		public byte section;

		public RenderInfo() {
		}

		public RenderInfo(IFlower flower, TileEntityFlower tile) {
			section = (byte) tile.getSection();
			primary = flower.getGenome().getPrimaryColor();
			secondary = flower.getGenome().getSecondaryColor();
			stem = flower.getGenome().getStemColor();
			age = (byte) flower.getAge();
			wilted = flower.isWilted();
			flowered = flower.hasFlowered();
			type = flower.getGenome().getType();
		}

		@Override
		public boolean equals(Object obj) {
			if (obj instanceof RenderInfo) {
				RenderInfo o = (RenderInfo) obj;
				return o.age == age && o.wilted == wilted && o.flowered == flowered && o.primary == primary && o.secondary == secondary && o.stem == stem && o.type == type;
			}
			return super.equals(obj);
		}
	}
}
